<?php

class Awardit_Antifraud_Model_Rule extends Mage_Core_Model_Abstract implements Awardit_Antifraud_ResolverInterface
{
    /** @var Awardit_Antifraud_Helper_Data */
    private $helper;

    /**
     * Initialize object.
     */
    protected function _construct(): void
    {
        $this->helper = Mage::helper('awardit_antifraud');
        $this->_init('awardit_antifraud/rule');
    }

    /**
     * If entry exists in database.
     * @return bool If entry exists in database.
     */
    public function exists(): bool
    {
        return !is_null($this->getId());
    }

    /**
     * Get mode.
     * @return string Mode of this instance.
     */
    public function getMode(): string
    {
        return (string)$this->getData('mode');
    }

    /**
     * Resolve rules for Quote or Order instance.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @return Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function getResolved(Mage_Core_Model_Abstract $model): Awardit_Antifraud_Model_Result
    {
        $collection = $this->getCollection();
        $collection->addFieldToFilter('store_id', ['in' => [(int)$model->getStoreId(), 0]]);
        $result = new Awardit_Antifraud_Model_Result();

        foreach ($collection as $rule) {
            $orders = Mage::getModel('awardit_antifraud/order')->getCollection();
            $select = $orders->getSelect();
            $select->reset();
            $select->from(['ao' => $orders->getResource()->getMainTable()], '');

            // Joins by rule type
            switch ($rule->getRuleType()) {
                case 'item_count':
                    $orders->join(['aoi' => 'order_item'], 'aoi.order_id=ao.order_id');
                    break;
            }

            $select->limit(1);
            if (!empty($rule->getStoreId())) {
                $select->where("ao.store_id = {$rule->getStoreId()}");
            }
            $select->where("ao.created_at >= DATE_SUB(NOW(), INTERVAL {$rule->getGroupInterval()} HOUR)");
            if ($states = Mage::getStoreConfig('awardit_antifraud/general/order_states', $model->getStoreId())) {
                $select->where(sprintf("ao.order_state IN (%s)", implode(',', array_map('json_encode', explode(',', $states)))));
            }

            if ($model instanceof Mage_Sales_Model_Order) {
                $select->where("ao.order_id != {$model->getId()}");
            }

            // Selects by group
            switch ($rule->getGroupType()) {
                case 'ip':
                    $ip = $this->helper->resolveIpAddr($model);
                    $select->where("ao.ip = '{$ip}'");
                    break;
                case 'email_user_hash':
                    $select->where("ao.email_user_hash = '{$this->helper->resolveEmailUserHash($model)}'");
                    break;
                case 'email_domain':
                    $select->where("ao.email_domain = '{$this->helper->resolveEmailDomain($model)}'");
                    break;
            }

            // Selects by rule type
            switch ($rule->getRuleType()) {
                case 'order_count':
                    $select->columns(new Zend_Db_Expr("(COUNT(order_id) + 1) AS acc_order_count"));
                    $select->having("acc_order_count >= {$rule->getRuleValue()}");
                    break;
                case 'order_value':
                    $total = $model->getBaseGrandTotal() * $model->getBaseToGlobalRate();
                    $select->columns(new Zend_Db_Expr("(SUM(order_value) + {$total}) AS acc_order_value"));
                    $select->having("acc_order_value >= {$rule->getRuleValue()}");
                    break;
                case 'item_count':
                    $qty = $model instanceof Mage_Sales_Model_Quote ? $model->getItemsSummaryQty() : $model->getTotalQtyOrdered();
                    $select->columns(new Zend_Db_Expr("(SUM(aoi.qty) + {$qty}) AS acc_item_count"));
                    $select->having("acc_item_count >= {$rule->getRuleValue()}");
                    break;
                case 'product_count':
                    $qty = $model instanceof Mage_Sales_Model_Quote ? 'qty' : 'qty_ordered';

                    $subselect = new Zend_Db_Select($orders->getResource()->getReadConnection());
                    $subselect->from(['aoi' => 'awardit_antifraud_order_item'], new Zend_Db_Expr("SUM(aoi.qty) AS aoi_product_count"));
                    $subselect->join(['ao' => 'awardit_antifraud_order'], "ao.order_id=aoi.order_id", '');
                    $subselect->where("aoi.sku=main_table.sku");
                    $subselect->where(implode(' ', $select->getPart(Zend_Db_Select::WHERE)));
                    $subselect->group("aoi.sku");

                    $orders = $model->getItemsCollection();
                    $select = $orders->getSelect();
                    $select->reset(Zend_Db_Select::COLUMNS);
                    $select->columns([
                        new Zend_Db_Expr("sku, SUM(main_table.{$qty}) AS mi_product_count"),
                        new Zend_Db_Expr("({$subselect}) AS aoi_product_count"),
                    ]);
                    $select->group("main_table.sku");
                    $select->having("mi_product_count + COALESCE(aoi_product_count, 0) >= {$rule->getRuleValue()}");
                    break;
            }

            // If match, apply rule
            if ($orders->fetchItem()) {
                $result->append($rule);
            }
        }

        // Return result accumulation
        return $result;
    }

    public function describe(): string
    {
        $modes = $this->helper->getModeOptions();
        $rtypes = $this->helper->getRuleTypeOptions();
        $gtypes = $this->helper->getGroupTypeOptions();
        $gintervals = $this->helper->getGroupIntervalOptions();
        return "Fraud {$modes[$this->getMode()]}: {$rtypes[$this->getRuleType()]} ≥ {$this->getRuleValue()} for {$gtypes[$this->getGroupType()]} in {$gintervals[$this->getGroupInterval()]}";
    }
}
