<?php

class Awardit_Antifraud_Model_Observer extends Mage_Core_Model_Abstract
{
    /**
     * Index order and items for future fraud evaluation.
     * @param Varien_Event_Observer $observer Observer with Order instance.
     */
    public function indexOrder(Varien_Event_Observer $observer): void
    {
        $order = $observer->getOrder();

        try {
            $af_order = Mage::getModel('awardit_antifraud/order');
            $af_order->indexOrder($order);

        } catch (Throwable $t) {
            // Log but don't break
            Mage::logException($t);
        }
    }

    /**
     * Run when customer inititates checkout. Antifraud module might prevent checkout from
     * completing, effectivly blocking customer from creating an order.
     * @param Varien_Event_Observer $observer Observer with Quote instance.
     * @throws Awardit_Antifraud_Exception If checkout should be denied.
     */
    public function beforeCheckout(Varien_Event_Observer $observer): void
    {
        $helper = Mage::helper('awardit_antifraud');
        $quote = $observer->getQuote();
        if (!$helper->isEnabled($quote->getStore())) {
            return;
        }

        $result = Mage::getModel('awardit_antifraud/blacklist')->getResolved($quote);
        $result->merge(Mage::getModel('awardit_antifraud/rule')->getResolved($quote));
        $quote->setAntifraudResult($result);

        switch ($result->getMode()) {
            case Awardit_Antifraud_ResolverInterface::ALLOW :
                // Always allowed, skip additional checks
                return;
            case Awardit_Antifraud_ResolverInterface::DENY:
                $helper->alertEmail($quote, $result);
                $helper->alertSlack($quote, $result);
                // Block checkout
                throw new Awardit_Antifraud_Exception($helper->getDenyMessage($quote->getStore()));
            case Awardit_Antifraud_ResolverInterface::HOLD:
            case Awardit_Antifraud_ResolverInterface::ALERT:
                // Don't block, but keep for later use
                break;
        }
    }

    /**
     * Run when customer completes checkout. Antifraud module might put order on hold.
     * @param Varien_Event_Observer $observer Observer with Quote and Order instance.
     */
    public function afterCheckout(Varien_Event_Observer $observer): void
    {
        $helper = Mage::helper('awardit_antifraud');
        $quote = $observer->getQuote();
        $order = $observer->getOrder();
        if (!$helper->isEnabled($order->getStore())) {
            return;
        }

        $result = $quote->getAntifraudResult();
        if (!$result) {
            $result = Mage::getModel('awardit_antifraud/blacklist')->getResolved($order);
            $result->merge(Mage::getModel('awardit_antifraud/rule')->getResolved($order));
        }

        switch ($result->getMode()) {
            case Awardit_Antifraud_ResolverInterface::ALLOW :
                // Always allowed, skip additional checks
                break;
            case Awardit_Antifraud_ResolverInterface::ALERT:
                // Alert but no other action
                $helper->alertEmail($order, $result);
                $helper->alertSlack($order, $result);
                break;
            case Awardit_Antifraud_ResolverInterface::DENY:
            case Awardit_Antifraud_ResolverInterface::HOLD:
                $helper->alertEmail($order, $result);
                $helper->alertSlack($order, $result);
                // Can't deny here, put on hold instead
                $order->setActionFlag(Mage_Sales_Model_Order::ACTION_FLAG_HOLD, true);
                if ($order->canHold()) {
                    $order->hold();
                }
                break;
            case '':
                return; // Not in status history
        }
        $order->addStatusHistoryComment("Checked by Antifraud: {$result->getMode()}");
        $order->save();
    }
}
