<?php

class Awardit_Antifraud_Helper_Data extends Mage_Core_Helper_Abstract
{
    // ----- Notifications ----------------------------------------------------------- //

    /**
     * Send email to admins when antifraud has evaluated Order or Quote.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to alert for.
     * @param Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function alertEmail(Mage_Core_Model_Abstract $model, Awardit_Antifraud_Model_Result $result): void
    {
        if (Mage::getStoreConfigFlag('system/smtp/disable')) {
            return; // Mail disabled
        }
        preg_match_all(
            '/([\w.-_]{1,}@[\w.-_]{1,}\.[\w]{2,})/',
            Mage::getStoreConfig('awardit_antifraud/alerts/alert_email', $model->getStore()),
            $matches
        );
        $receivers = array_shift($matches);
        if (empty($receivers)) {
            return; // No-one to send to
        }
        $sender = Mage::getStoreConfig('awardit_antifraud/alerts/alert_context', $model->getStore()) ?: $this->__('Magento Antifraud module');

        $body = array_merge([
            $this->__('An order has been evaluated for fraud with result: %s.', $result->getMode()),
            '',
            $this->__('Check result:'),
        ], $result->getDescriptionList('• '), [
            '',
            $this->__('Summary:'),
        ], $this->getSaleInfo($model));

        $mail = new Zend_Mail('UTF-8');
        $mail->setBodyText(implode("\r\n", $body))
            ->setFrom('noreply@awardit.com', $sender)
            ->setSubject($this->__('Antifraud alert'));
        foreach ($receivers as $receiver) {
            $mail->addTo($receiver);
        }
        $mail->send();
    }

    /**
     * Send to Slach channel when antifraud has evaluated Order or Quote.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to alert for.
     * @param Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function alertSlack(Mage_Core_Model_Abstract $model, Awardit_Antifraud_Model_Result $result): void
    {
        $conversation = Mage::getStoreConfig('awardit_antifraud/general/alert_slack', $model->getStore());
        if (empty($conversation)) {
            return; // No-one to send to
        }
        $sender = Mage::getStoreConfig('awardit_antifraud/alerts/alert_context', $model->getStore()) ?: $this->__('Magento Antifraud module');

        $body = array_merge([
            $this->__('An order has been evaluated for fraud with result: *%s*.', $result->getMode()),
            '',
            $this->__('*Check result:*'),
        ], $result->getDescriptionList('• '), [
            '',
            $this->__('*Summary:*'),
        ], $this->getSaleInfo($model));

        $slack = Mage::helper('awardit_slack');
        $slack->postInChannel($conversation, 'Testar', [
            [
                'type' => 'header',
                'text' => [
                    'type' => 'plain_text',
                    'text' => $this->__('Antifraud alert'),
                ]
            ],
            [
                'type' => 'section',
                'text' => [
                    'type' => 'mrkdwn',
                    'text' => implode("\n", $body),
                ]
            ],
            [
                'type' => 'context',
                'elements' => [
                    [
                        'type' => 'mrkdwn',
                        'text' => $sender,
                    ],
                ]
            ],
        ]);
    }

    /**
     * Message to use when checkout is denied.
     * @param Mage_Core_Model_Store|null $store Store to use.
     * @return string Deny message.
     */
    public function getDenyMessage(Mage_Core_Model_Store $store = null): string
    {
        $message = Mage::getStoreConfig('awardit_antifraud/general/deny_message', $store);
        return $message ?: $this->__('Order can not be placed.');
    }

    /**
     * Get sale summary for Quote/Order.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to get info on.
     * @return array<string> Sale info.
     */
    private function getSaleInfo(Mage_Core_Model_Abstract $model): array
    {
        return [
            $model instanceof Mage_Sales_Model_Order
                ? $this->__('Order: %s', $model->getIncrementId())
                : $this->__('Quote: %s', $model->getId()),
            $this->__('Email: %s', $model->getCustomerEmail()),
            $this->__('IP-address: %s', $this->resolveIpAddr($model)),
            $this->__('Name: %s', $model->getCustomerName()),
        ];
    }


    // ----- Verifiers --------------------------------------------------------------- //

    /**
     * If antifraud evaluation should be applied.
     * @param Mage_Core_Model_Store|null $store Store to use.
     * @return bool If enabled.
     */
    public function isEnabled(Mage_Core_Model_Store $store = null): bool
    {
        return (bool)Mage::getStoreConfig('awardit_antifraud/general/enabled', $store);
    }

    /**
     * Verifies that used currencies have defined currency rate to base currency.
     * @return bool True if rates are correctly defined.
     */
    public function hasCurrencyRates(): bool
    {
        $currency = Mage::getModel('directory/currency');
        $base_currency = Mage::app()->getBaseCurrencyCode();
        $used_currencies = $currency->getConfigAllowCurrencies();
        $currency_rates = $currency->getCurrencyRates($base_currency, $used_currencies);
        return count($currency_rates) >= count($used_currencies);
    }


    // ----- Resolvers --------------------------------------------------------------- //

    /**
     * Resolve IP-adddress.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Resolve source.
     * @return string IP-adddress.
     */
    public function resolveIpAddr(Mage_Core_Model_Abstract $model): string
    {
        $ip = $model->getData('x_forwarded_for') ?? $model->getData('remote_ip') ?? '127.0.0.1';
        if (empty($ip)) {
            return '';
        }

        // Can contain comma separated list of IP-addresses.
        $ips = array_filter(array_map('trim', explode(',', $ip)));

        return array_pop($ips);
    }

    /**
     * Resolve Email user as hash.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Resolve source.
     * @return string Email user hash.
     */
    public function resolveEmailUserHash(Mage_Core_Model_Abstract $model): string
    {
        $email = trim($model->getCustomerEmail());
        if (empty($email)) {
            return hash('sha256', 'empty');
        }
        $parts = explode('@', trim($email));
        if (count($parts) < 2) {
            return hash('sha256', trim($email));
        }
        return hash('sha256', trim($parts[0]));
    }

    /**
     * Resolve Email domain.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Resolve source.
     * @return string Email domain.
     */
    public function resolveEmailDomain(Mage_Core_Model_Abstract $model): string
    {
        $email = trim($model->getCustomerEmail());
        if (empty($email)) {
            return 'faulty';
        }
        $parts = explode('@', trim($email));
        if (count($parts) < 2) {
            return 'faulty';
        }
        return trim($parts[1]);
    }


    // ----- Options ----------------------------------------------------------------- //

    /**
     * Get Blacklist types.
     * @return array<array-key, string> Types.
     */
    public function getTypeOptions(): array
    {
        return [
            'email' => $this->__('Email'),
            'ip' => $this->__('IP-address'),
        ];
    }

    /**
     * Get Fraud rule group types.
     * @return array<array-key, string> Group types.
     */
    public function getGroupTypeOptions(): array
    {
        return [
            'email_user_hash' => $this->__('Email username'),
            'email_domain' => $this->__('Email domain'),
            'ip' => $this->__('IP-address'),
        ];
    }

    /**
     * Get Fraud rule rule types.
     * @return array<array-key, string> Rule types.
     */
    public function getRuleTypeOptions(): array
    {
        return [
            'order_count' => $this->__('Order count'),
            'order_value' => $this->__('Order value (%s)', Mage::app()->getBaseCurrencyCode()),
            'item_count' => $this->__('Item count'),
            'product_count' => $this->__('Product count'),
        ];
    }

    /**
     * Get Fraud rule intervals.
     * @return array<array-key, string> Intervals.
     */
    public function getGroupIntervalOptions(): array
    {
        return [
            0 => $this->__('Now'),
            1 => $this->__('1 hour'),
            2 => $this->__('2 hours'),
            4 => $this->__('4 hours'),
            12 => $this->__('12 hours'),
            24 => $this->__('1 day'),
            48 => $this->__('2 days'),
            168 => $this->__('1 week'),
            720 => $this->__('1 month'),
            8760 => $this->__('1 year'),
        ];
    }

    /**
     * Get modes.
     * @return array<array-key, string> Modes.
     */
    public function getModeOptions(): array
    {
        return [
            Awardit_Antifraud_ResolverInterface::ALLOW => $this->__('Allow'),
            Awardit_Antifraud_ResolverInterface::ALERT => $this->__('Alert'),
            Awardit_Antifraud_ResolverInterface::HOLD => $this->__('Hold'),
            Awardit_Antifraud_ResolverInterface::DENY => $this->__('Deny'),
        ];
    }

    /**
     * Get stores.
     * @return array<array-key, string> Stores.
     */
    public function getStoreOptions(): array
    {
        $stores = [0 => "{$this->__('Global')} (0)"];
        foreach (Mage::app()->getStores(false, true) as $store) {
            $stores[$store->getId()] = "{$store->getName()} ({$store->getId()})";
        }
        return $stores;
    }
}
