<?php

class Awardit_Antifraud_Model_Observer extends Mage_Core_Model_Abstract
{
    /** @var bool */
    private $processing = false;

    /**
     * Index order and items for future fraud evaluation.
     * @param Varien_Event_Observer $observer Observer with Order instance.
     */
    public function indexOrder(Varien_Event_Observer $observer): void
    {
        $order = $observer->getOrder();
        try {
            $af_order = Mage::getModel('awardit_antifraud/order');
            $af_order->indexOrder($order);
        } catch (Throwable $t) {
            // Log but don't break
            Mage::logException($t);
        }
    }

    /**
     * Run when customer inititates checkout. Antifraud module might prevent checkout from
     * completing, effectivly blocking customer from creating an order.
     * @param Varien_Event_Observer $observer Observer with Quote instance.
     * @throws Awardit_Antifraud_Exception If checkout should be denied.
     */
    public function beforeCheckout(Varien_Event_Observer $observer): void
    {
        $this->processing = true;
        $helper = Mage::helper('awardit_antifraud');
        $quote = $observer->getQuote();
        if (!$helper->isEnabled($quote->getStore())) {
            return;
        }

        if ($quote->getAntifraudResult()) {
            return; // Already resolved
        }

        $result = new Awardit_Antifraud_Model_Result();
        $result->merge(Mage::getModel('awardit_antifraud/blacklist')->getResolved($quote));
        $result->merge(Mage::getModel('awardit_antifraud/rule')->getResolved($quote));
        $quote->setAntifraudResult($result);

        $mode = $result->getMode() ?: 'ok';
        $helper->log("Antifraud: Quote checkout {$quote->getId()}: {$mode}.");
        switch ($result->getMode()) {
            case Awardit_Antifraud_ResolverInterface::ALLOW:
                // Always allowed, skip additional checks
                return;
            case Awardit_Antifraud_ResolverInterface::DENY:
                $helper->alertEmail($quote, $result);
                $helper->alertSlack($quote, $result);
                // Block checkout
                throw new Awardit_Antifraud_Exception($helper->getDenyMessage($quote->getStore()));
            case Awardit_Antifraud_ResolverInterface::HOLD:
            case Awardit_Antifraud_ResolverInterface::ALERT:
                // Don't block, but keep for later use
                break;
        }
    }

    /**
     * Run when customer completes checkout. Antifraud module might put order on hold.
     * @param Varien_Event_Observer $observer Observer with Quote and Order instance.
     */
    public function afterCheckout(Varien_Event_Observer $observer): void
    {
        $this->processing = true;
        $helper = Mage::helper('awardit_antifraud');
        /** @var Mage_Sales_Model_Quote $order */
        $quote = $observer->getQuote();
        /** @var Mage_Sales_Model_Order $order */
        $order = $observer->getOrder();
        if (!$helper->isEnabled($order->getStore())) {
            return;
        }

        $result = $quote->getAntifraudResult() ?: new Awardit_Antifraud_Model_Result();
        $result->merge(Mage::getModel('awardit_antifraud/blacklist')->getResolved($order));
        $result->merge(Mage::getModel('awardit_antifraud/rule')->getResolved($order));

        $mode = $result->getMode() ?: 'ok';
        $helper->log("Antifraud: Order checkout {$order->getIncrementId()}: {$mode}.");
        switch ($result->getMode()) {
            case Awardit_Antifraud_ResolverInterface::ALLOW:
                // Always allowed, skip additional checks
                break;
            case Awardit_Antifraud_ResolverInterface::ALERT:
                // Alert but no other action
                $helper->alertEmail($order, $result);
                $helper->alertSlack($order, $result);
                break;
            case Awardit_Antifraud_ResolverInterface::DENY:
            case Awardit_Antifraud_ResolverInterface::HOLD:
                // Can't deny here, put on hold instead
                if ($helper->isHoldable($order)) {
                    $helper->setHoldStatus($order);
                }
                $helper->alertEmail($order, $result);
                $helper->alertSlack($order, $result);
                break;
            default:
                return; // Not in status history
        }

        $modes = $helper->getModeOptions();
        $order->addStatusHistoryComment(
            $helper->__(
                "Checked by Antifraud: <b>%s</b>.<br>%s",
                $modes[$result->getMode()],
                implode('<br>', $result->getDescriptionList('• '))
            ),
            false
        );
        $order->save();
    }

    /**
     * Run when order status changes. Antifraud module might put order on hold.
     * @param Varien_Event_Observer $observer Observer with Quote and Order instance.
     */
    public function afterStateChange(Varien_Event_Observer $observer): void
    {
        if ($this->processing) {
            return; // Only run when state changed as a separate request
        }
        $this->processing = true;
        $helper = Mage::helper('awardit_antifraud');
        $order = $observer->getOrder();
        $quote = $observer->getQuote() ?: $order->getQuote();

        if (!$helper->isEnabled($order->getStore())) {
            return;
        }
        if (!$helper->changedToHoldable($order)) {
            return; // Can't do anything
        }

        $result = $quote ? $quote->getAntifraudResult() : null;
        if (!$result) {
            $result = new Awardit_Antifraud_Model_Result();
        }
        $result->merge(Mage::getModel('awardit_antifraud/blacklist')->getResolved($order));
        $result->merge(Mage::getModel('awardit_antifraud/rule')->getResolved($order));

        $mode = $result->getMode() ?: 'ok';
        $helper->log("Antifraud: Order state change {$order->getIncrementId()}: {$mode}.");
        switch ($result->getMode()) {
            case Awardit_Antifraud_ResolverInterface::ALLOW:
            case Awardit_Antifraud_ResolverInterface::ALERT:
                // All done already
                return;
            case Awardit_Antifraud_ResolverInterface::DENY:
            case Awardit_Antifraud_ResolverInterface::HOLD:
                // Can't deny here, put on hold instead
                $helper->setHoldStatus($order);
                $helper->alertEmail($order, $result);
                $helper->alertSlack($order, $result);
                break;
            default:
                return; // Not action required
        }

        $modes = $helper->getModeOptions();
        $order->addStatusHistoryComment(
            $helper->__(
                "Holded by Antifraud: <b>%s</b>.<br>%s",
                $modes[$result->getMode()],
                implode('<br>', $result->getDescriptionList('• '))
            ),
            false
        );
        $order->save();
    }
}
