<?php

/**
 * @psalm-type OrderOrQuote = Mage_Sales_Model_Order|Mage_Sales_Model_Quote|Mage_Core_Model_Abstract
 * @method string getMode()
 * @method void setMode(string $mode)
 * @method int getStoreId()
 * @method void setStoreId(int $storeId)
 * @method string getType()
 * @method void setType(string $type)
 * @method float|null getValue()
 * @method void setValue(float|null $value)
 */
class Awardit_Antifraud_Model_Blacklist extends Mage_Core_Model_Abstract implements Awardit_Antifraud_ResolverInterface
{
    /** @var Awardit_Antifraud_Helper_Data */
    private $helper;

    public function __construct()
    {
        $this->helper = Mage::helper('awardit_antifraud');
        parent::__construct();
    }

    /**
     * Initialize object.
     */
    protected function _construct(): void
    {
        $this->_init('awardit_antifraud/blacklist');
    }

    /**
     * If entry exists in database.
     * @return bool If entry exists in database.
     */
    public function exists(): bool
    {
        return !is_null($this->getId());
    }

    /**
     * Get mode.
     * @return string Mode of this instance.
     */
    public function getMode(): string
    {
        return (string)$this->getData('mode');
    }

    /**
     * Resolve mode for Quote or Order instance.
     * @param OrderOrQuote $model Instance to resolve.
     * @return Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function getResolved(Mage_Core_Model_Abstract $model): Awardit_Antifraud_Model_Result
    {
        $ip = $this->helper->resolveIpAddr($model);
        $email = trim($model->getCustomerEmail() ?? '');
        $riskRecommendation = $this->helper->resolveRiskRecommendation($model);
        $result = new Awardit_Antifraud_Model_Result();

        /** @var Awardit_Antifraud_Model_Resource_Blacklist_Collection $collection */
        $collection = $this->getCollection();
        $select = $collection->getSelect();
        $select->where("(store_id = ? OR store_id = 0)", $model->getStoreId());
        $orWhere = [
            $select->getAdapter()->quoteInto("(type = 'ip' AND ? LIKE value)", $ip),
            $select->getAdapter()->quoteInto("(type = 'email' AND ? LIKE value)", $email),
            $select->getAdapter()->quoteInto("(type = 'risk_recommendation' AND ? LIKE value)", $riskRecommendation)
        ];
        $select->where(implode(' OR ', $orWhere));
        $select->order(new Zend_Db_Expr("mode='allow' DESC, mode='deny' DESC, mode='hold' DESC"));
        $select->limit(1);

        // If match, apply blacklist
        /** @var Awardit_Antifraud_Model_Blacklist|null $blacklist */
        $blacklist = $collection->fetchItem();
        if ($blacklist) {
            $result->append($blacklist);
        }
        return $result;
    }

    /**
     * Describe application.
     * @param boolean $markdown
     * @return string Description.
     */
    public function describe(bool $markdown = false): string
    {
        $modes = $this->helper->getModeOptions();
        $types = $this->helper->getTypeOptions();
        return "Blacklist {$modes[$this->getMode()]}: {$types[$this->getType()]} = {$this->getValue()}";
    }
}
