<?php

declare(strict_types=1);

use Crossroads\Magento\Test\Integration\MagentoManager;
use PHPUnit\Framework\TestCase;

class AntifraudTest extends TestCase
{
    /** @var Mage_Core_Model_Store */
    private $store;
    /** @var Mage_Catalog_Model_Product */
    private $product;

    public function setUp(): void
    {
        MagentoManager::reset();
        MagentoManager::init('default');

        $config = new Mage_Core_Model_Config();
        $config->saveConfig('awardit_antifraud/general/enabled', '1', 'default', 0);
        $config->saveConfig('awardit_antifraud/general/order_states', 'new,pending_payment,processing,complete,closed,canceled,holded,payment_review', 'default', 0);
        Mage::getConfig()->loadModules()->loadDb()->saveCache(); // Reload config cache
        $this->store = Mage::app()->getStore();
        $this->setUpProduct();;
    }

    public function tearDown(): void
    {
        MagentoManager::logQueries();
    }


    // ----- Blacklist tests --------------------------------------------------------- //

    // Should check but take no action.
    public function testBlacklistAlert(): void
    {
        $blacklist = Mage::getModel('awardit_antifraud/blacklist');
        $blacklist->setData([
            'store_id' => $this->store->getId(),
            'type' => 'email',
            'value' => 'test@awardit-antifraud.com',
            'mode' => 'alert',
            'created_at' => date('c'),
        ])->save();

        // Setup, run and check quote

        $quote = $this->getQuote();
        $observer = new Varien_Event_Observer();
        $observer->setQuote($quote);
        $handler = new Awardit_Antifraud_Model_Observer();
        $handler->beforeCheckout($observer);
        $result = $quote->getAntifraudResult();

        $this->assertInstanceof(Awardit_Antifraud_Model_Result::class, $result);
        $this->assertEquals('alert', $result->getMode());
        $this->assertCount(1, $result->getResolved());

        // Setup, run and check order

        $service = new Mage_Sales_Model_Service_Quote($quote);
        $order = $service->submit();
        $observer->setOrder($order);
        $handler->afterCheckout($observer);

        $this->assertEquals('pending', $order->getStatus());

        // Check indexing

        $index = Mage::getModel('awardit_antifraud/order');
        $index->load($order->getId());

        $this->assertEquals($order->getIncrementId(), $index->getIncrementId());
        $this->assertEquals('awardit-antifraud.com', $index->getEmailDomain());
    }

    // Allow checkout but set order on hold.
    public function testBlacklistHold(): void
    {
        $blacklist = Mage::getModel('awardit_antifraud/blacklist');
        $blacklist->setData([
            'store_id' => $this->store->getId(),
            'type' => 'email',
            'value' => 'test@awardit-antifraud.com',
            'mode' => 'hold',
            'created_at' => date('c'),
        ])->save();

        // Setup, run and check quote

        $quote = $this->getQuote();
        $observer = new Varien_Event_Observer();
        $observer->setQuote($quote);
        $handler = new Awardit_Antifraud_Model_Observer();
        $handler->beforeCheckout($observer);
        $result = $quote->getAntifraudResult();

        $this->assertInstanceof(Awardit_Antifraud_Model_Result::class, $result);
        $this->assertEquals('hold', $result->getMode());
        $this->assertCount(1, $result->getResolved());

        // Setup, run and check order

        $service = new Mage_Sales_Model_Service_Quote($quote);
        $order = $service->submit();
        $observer->setOrder($order);
        $handler->afterCheckout($observer);

        $this->assertEquals('holded', $order->getStatus());
    }

    // Deny checkout.
    public function testBlacklistDeny(): void
    {
        $blacklist = Mage::getModel('awardit_antifraud/blacklist');
        $blacklist->setData([
            'store_id' => $this->store->getId(),
            'type' => 'email',
            'value' => 'test@awardit-antifraud.com',
            'mode' => 'deny',
            'created_at' => date('c'),
        ])->save();

        // Setup, run and check quote

        $quote = $this->getQuote();
        $observer = new Varien_Event_Observer();
        $observer->setQuote($quote);
        $handler = new Awardit_Antifraud_Model_Observer();

        $this->expectException(Awardit_Antifraud_Exception::class);
        $this->expectExceptionMessage('Order can not be placed.');
        $handler->beforeCheckout($observer);
    }

    // Allow checkout and order despite previous entries.
    public function testBlacklistAllow(): void
    {
        $blacklist = Mage::getModel('awardit_antifraud/blacklist');
        $blacklist->setData([
            'store_id' => $this->store->getId(),
            'type' => 'email',
            'value' => 'test@awardit-antifraud.com',
            'mode' => 'allow',
            'created_at' => date('c'),
        ])->save();

        // Setup, run and check quote

        $quote = $this->getQuote();
        $observer = new Varien_Event_Observer();
        $observer->setQuote($quote);
        $handler = new Awardit_Antifraud_Model_Observer();
        $handler->beforeCheckout($observer);
        $result = $quote->getAntifraudResult();

        $this->assertInstanceof(Awardit_Antifraud_Model_Result::class, $result);
        $this->assertEquals('allow', $result->getMode());
        $this->assertCount(1, $result->getResolved());

        // Setup, run and check order

        $service = new Mage_Sales_Model_Service_Quote($quote);
        $order = $service->submit();
        $observer->setOrder($order);
        $handler->afterCheckout($observer);

        $this->assertEquals('pending', $order->getStatus());
    }

    public function testRules(): void
    {
        // Make all rules trigger.

        $rule = Mage::getModel('awardit_antifraud/rule');
        $rule->setData([
            'store_id' => $this->store->getId(),
            'rule_type' => 'order_count',
            'rule_value' => 4,
            'group_type' => 'email_domain',
            'group_interval' => 2,
            'mode' => 'hold',
            'created_at' => date('c'),
        ])->save();

        $rule = Mage::getModel('awardit_antifraud/rule');
        $rule->setData([
            'store_id' => $this->store->getId(),
            'rule_type' => 'order_value',
            'rule_value' => 0,
            'group_type' => 'email_domain',
            'group_interval' => 2,
            'mode' => 'alert',
            'created_at' => date('c'),
        ])->save();

        $rule = Mage::getModel('awardit_antifraud/rule');
        $rule->setData([
            'store_id' => $this->store->getId(),
            'rule_type' => 'item_count',
            'rule_value' => 0,
            'group_type' => 'email_domain',
            'group_interval' => 2,
            'mode' => 'deny',
            'created_at' => date('c'),
        ])->save();

        $rule = Mage::getModel('awardit_antifraud/rule');
        $rule->setData([
            'store_id' => $this->store->getId(),
            'rule_type' => 'product_count',
            'rule_value' => 10,
            'group_type' => 'email_domain',
            'group_interval' => 2,
            'mode' => 'allow',
            'created_at' => date('c'),
        ])->save();

        // Setup, run and check quote

        $quote = $this->getQuote();
        $observer = new Varien_Event_Observer();
        $observer->setQuote($quote);
        $handler = new Awardit_Antifraud_Model_Observer();
        $handler->beforeCheckout($observer);
        $result = $quote->getAntifraudResult();
        $resolved = $result->getResolved();

        $this->assertInstanceof(Awardit_Antifraud_Model_Result::class, $result);
        $this->assertEquals('allow', $result->getMode());
        $this->assertCount(5, $resolved);

        $this->assertInstanceof(Awardit_Antifraud_Model_Rule::class, $resolved[0]);
        $this->assertEquals('item_count', $resolved[0]->getRuleType());
        $this->assertEquals('deny', $resolved[0]->getMode());

        $this->assertInstanceof(Awardit_Antifraud_Model_Rule::class, $resolved[1]);
        $this->assertEquals('order_count', $resolved[1]->getRuleType());
        $this->assertEquals('hold', $resolved[1]->getMode());

        $this->assertInstanceof(Awardit_Antifraud_Model_Rule::class, $resolved[2]);
        $this->assertEquals('order_value', $resolved[2]->getRuleType());
        $this->assertEquals('alert', $resolved[2]->getMode());

        $this->assertInstanceof(Awardit_Antifraud_Model_Blacklist::class, $resolved[3]);
        $this->assertEquals('allow', $resolved[3]->getMode());

        $this->assertInstanceof(Awardit_Antifraud_Model_Rule::class, $resolved[4]);
        $this->assertEquals('product_count', $resolved[4]->getRuleType());
        $this->assertEquals('allow', $resolved[4]->getMode());

        // Setup, run and check order

        $service = new Mage_Sales_Model_Service_Quote($quote);
        $order = $service->submit();
        $observer->setOrder($order);
        $handler->afterCheckout($observer);

        $this->assertEquals('pending', $order->getStatus());
    }


    // ----- Helper methods ---------------------------------------------------------- //

    private function setUpProduct(): void
    {
        $key = rand(1000, 9999);
        $this->product = Mage::getModel('catalog/product');
        $this->product->fromArray([
            'store_id' => $this->store->getId(),
            'sku' => "sku-{$key}",
            'url_key' => "url-{$key}",
            'attribute_set_id' => $this->product->getDefaultAttributeSetId(),
            'type_id' => 'virtual',
        ])->save();
    }

    private function getQuote(): Mage_Sales_Model_Quote
    {
        $quote = Mage::getSingleton('checkout/session')->getQuote();
        $quote->setAntifraudResult(null); // Reset fraud evaluation
        $quote->setIsSuperMode(true); // Bypass some checks
        $address = Mage::getModel('sales/quote_address')->addData([
            'firstname' => 'Test',
            'lastname' => 'Testmann',
            'street' => 'Testgatan 12',
            'postcode' => '12345',
            'city' => 'Ankeborg',
            'country_id' => 'SE',
            'telephone' => '011 223344',
        ]);
        $quote->setBillingAddress($address);
        $quote->setShippingAddress($address);
        $quote->setQuoteCurrencyCode('SEK');
        $quote->setCustomerEmail('test@awardit-antifraud.com');
        $options = new Varien_Object(['qty' => 10]);
        $quote->addProduct($this->product, $options);
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();
        $payment = Mage::getModel('sales/quote_payment')->setMethod('free');
        $payment->setQuote($quote)->save();
        $quote->setPayment($payment);
        $quote->save();
        return $quote;
    }
}
