<?php

use Awardit_Altapay_Model_Checkout_Endpoint_BaseApi as BaseApi;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * @extends Awardit_Altapay_Model_Checkout_Endpoint_BaseApi<Awardit_Altapay_Model_Checkout_Type_AuthenticateRequest|null>
 */
class Awardit_Altapay_Model_Checkout_Endpoint_Authentication extends BaseApi
{
    /** @var Awardit_Altapay_Model_Checkout_Configuration */
    protected $config;

    /** @var Awardit_Altapay_Model_Checkout_Type_AuthenticateRequest|null */
    protected $body;

    /**
     * @param Awardit_Altapay_Model_Checkout_Configuration $config
     */
    public function __construct(
        Awardit_Altapay_Model_Checkout_Configuration $config,
        ?Awardit_Altapay_Model_Checkout_Type_AuthenticateRequest $requestBody = null
    ) {
        $this->config = $config;
        $this->body = $requestBody;
    }

    public function getMethod(): string
    {
        return 'POST';
    }

    public function getUri(): string
    {
        return '/checkout/v1/api/authenticate';
    }

    public function getHeaders(array $baseHeaders = []): array
    {
        $headers = ['Content-Type' => 'application/json'];

        // this endpoint requires HTTP basic authentication
        if (!empty($this->config->getUsername()) || !(empty($this->config->getPassword()))) {
            $headers['Authorization'] = sprintf(
                'Basic %s',
                base64_encode(sprintf('%s:%s', $this->config->getUsername(), $this->config->getPassword()))
            );
        }
        return array_merge($baseHeaders, $headers);
    }

    public function getBody(SerializerInterface $serializer): string
    {
        return $serializer->serialize($this->body, 'json');
    }

    /**
     * @param ResponseInterface $response
     * @param SerializerInterface $serializer
     * @return null|Awardit_Altapay_Model_Checkout_Type_TokenResponse
     */
    public function parseResponse(ResponseInterface $response, SerializerInterface $serializer)
    {
        $status = $response->getStatusCode();
        $contentType = $response->hasHeader('Content-Type') ? current($response->getHeader('Content-Type')) : null;
        $body = (string) $response->getBody();

        if ($status === 200 && $contentType === 'application/json') {
            return $serializer->deserialize($body, 'Awardit_Altapay_Model_Checkout_Type_TokenResponse', 'json');
        }
        if ($status === 400 && $contentType === 'application/json') {
            throw new Awardit_Altapay_Model_Checkout_Exception_BadRequestException(
                'Wrong authenticate request body.',
                $serializer->deserialize($body, 'Awardit_Altapay_Model_Checkout_Type_ErrorResponse', 'json')
            );
        }
        if ($status === 401 && $contentType === 'application/json') {
            throw new Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException(
                'Authentication issue - missing or wrong credentials.',
                $serializer->deserialize($body, 'Awardit_Altapay_Model_Checkout_Type_ErrorResponse', 'json')
            );
        }
    }
}
