<?php

use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Promise\Create;
use GuzzleHttp\Promise\PromiseInterface;
use Psr\Http\Message\MessageInterface;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;

/**
 * Guzzle logger middleware
 *
 * Inpsired by https://github.com/placetopay-org/guzzle-logger
 */
class Awardit_Altapay_Model_Api_LoggerMiddleware
{
    private LoggerInterface $logger;

    /**
     * @param LoggerInterface $logger
     */
    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * @param callable $handler
     * @return callable
     */
    public function __invoke(callable $handler): callable
    {
        return function (RequestInterface $request, array $options) use ($handler): PromiseInterface {
            return $handler($request, $options)
                ->then(
                    $this->onFulfilled($request),
                    $this->onRejected($request)
                );
        };
    }

    /**
     * @param RequestInterface $request
     * @return callable
     */
    private function onFulfilled(RequestInterface $request): callable
    {
        return function (ResponseInterface $response) use ($request) {
            $this->log($request, $response, null);

            return $response;
        };
    }

    /**
     * @param RequestInterface $request
     * @return callable
     */
    private function onRejected(RequestInterface $request): callable
    {
        return function (\Exception $reason) use ($request) {
            if ($reason instanceof RequestException && $reason->hasResponse() === true) {
                $this->log($request, $reason->getResponse(), null);

                return Create::rejectionFor($reason);
            }

            $this->log($request, null, $reason);

            return Create::rejectionFor($reason);
        };
    }

    /**
     * @param RequestInterface $request
     * @param ResponseInterface|null $response
     * @param Throwable|null $exception
     * @return void
     */
    private function log(
        RequestInterface $request,
        ?ResponseInterface $response = null,
        ?Throwable $exception = null
    ): void {
        if ($exception !== null) {
            $this->logger->error('Requested Altapay API: {method} {uri}', [
                'method' => $request->getMethod(),
                'uri' => $request->getUri()->__toString(),
                'exception' => $exception,
            ]);
        }

        $requestContext = $this->getRequestContext($request);

        if ($response !== null) {
            $status = $response->getStatusCode();
            $responseContext = $this->getResponseContext($response);
        } else {
            $status = 200;
            $responseContext = [];
        }

        if ($status >= 500) {
            $level = LogLevel::ERROR;
        } elseif ($status >= 400) {
            $level = LogLevel::WARNING;
        } else {
            $level = LogLevel::INFO;
        }

        $this->logger->log(
            $level,
            sprintf(
                'Requested Altapay API: %s %s %s',
                $request->getMethod(),
                $request->getUri()->__toString(),
                $response ? $response->getReasonPhrase() : ''
            ),
            array_merge($requestContext, $responseContext)
        );
    }

    /**
     * @param RequestInterface $request
     * @return array
     */
    private function getRequestContext(RequestInterface $request): array
    {
        return [
            'request' => array_filter([
                'url' => $request->getUri()->__toString(),
                'body' => $request->getBody()->getSize() > 0 ? $this->formatBody($request) : null,
                'method' => $request->getMethod(),
            ]),
        ];
    }

    /**
     * @param ResponseInterface $response
     * @return array
     */
    private function getResponseContext(ResponseInterface $response): array
    {
        return [
            'response' => array_filter([
                'body' => $this->formatBody($response),
                'code' => $response->getStatusCode(),
                'message' => $response->getReasonPhrase(),
            ]),
        ];
    }

    /**
     * @param MessageInterface $response
     * @return array|string
     */
    private function formatBody(MessageInterface $response)
    {
        $body = $response->getBody()->__toString();

        $json = json_decode($body, true);

        if (! empty($json)) {
            return $json;
        }

        if (strlen($body) === 0) {
            return 'Failed empty body';
        }

        return 'Failed to decode JSON from body: ' . self::bodySummary($body);
    }

    /**
     * @param string $body
     * @return string
     */
    private static function bodySummary(string $body): string
    {
        $size = strlen($body);

        $summary = mb_substr($body, 0, 120);
        if ($size > 120) {
            $summary .= ' (truncated...)';
        }

        return $summary;
    }
}
