<?php

namespace AltapayTests;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use GuzzleHttp\Psr7\Response;
use Mage;
use Mage_Core_Model_Config;
use Mage_Sales_Model_Quote;
use PHPUnit\Framework\TestCase;
use Varien_Object;

class CallbackTest extends TestCase
{
    public function setUp(): void
    {
        MagentoManager::reset();
        MagentoManager::init('default');

        $config = new Mage_Core_Model_Config();
        $config->saveConfig('payment/altapay/active', '1', 'default', 0);
        $config->saveConfig('payment/altapay/username', 'irrelevant', 'default', 0);
        $config->saveConfig('payment/altapay/password', 'irrelevant', 'default', 0);
        $config->saveConfig('payment/altapay/host', 'https://testgateway.altapaysecure.com', 'default', 0);
        Mage::app()->baseInit([]);
        Mage::getConfig()->loadModules()->loadDb()->saveCache(); // Reload config cache
    }

    protected function createQuote(): Mage_Sales_Model_Quote
    {
        // Create a mock and queue two responses.
        $headers = ['Content-Type' => 'application/json'];
        $responses = [
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
            // @todo: These are just put here, actual data should be checked
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
        ];

        $requests = [];
        Mage::register('_helper/awardit_altapay', MockHelper::getHelper($responses, $requests));

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();
        $address = [
            "firstname"  => "Jane",
            "lastname"   => "Doe",
            "street"     => "Alfagatan 1",
            "city"       => "Mölndal",
            "postcode"   => "43147",
            "country_id" => "SE",
            "telephone"  => "07001234567",
        ];

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData($address));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData($address));
        $quote->addProduct($product, $request);
        $quote->setIsActive(1);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->getPayment()->importData([
            "method" => "altapay",
            "additional_information" => [
                "method" => "altapay",
                "sessionId" => "66175043-6025-4453-b142-66320cbaaa09",
                "checkoutToken" => "irrelevant",
            ],
        ]);
        $quote->save();

        // Finalize Quote
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->reserveOrderId();
        $quote->save();

        return $quote;
    }

    public function testNotificationCallback(): void
    {
        $quote = $this->createQuote();
        // Create Magento Order
        $service = Mage::getModel('sales/service_quote', $quote);
        $service->submitAll();
        $order = $service->getOrder();
        $orderId = $order->getIncrementId();
        // Close Quote
        $quote->setIsActive(0);
        $quote->save();

        $body = file_get_contents(__DIR__ . '/fixtures/notification.json');
        $body = preg_replace('|<ORDERID>|', $orderId, $body);
        $body = preg_replace('|<QUOTEID>|', (string)$quote->getId(), $body);

        $response = MagentoManager::runRequest(new Request("POST /Awardit_Altapay/callback/notification", [
            "Content-Type" => "application/json",
        ], $body), 'default');
        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
    }

    public function testVerifyCallback(): void
    {
        $quote = $this->createQuote();
        /** @var string */
        $quoteId = $quote->getId();
        $body = file_get_contents(__DIR__ . '/fixtures/notification.json');

        $body = preg_replace('|<ORDERID>|', $quote->getReservedOrderId(), $body);
        $body = preg_replace('|<QUOTEID>|', (string)$quote->getId(), $body);

        $response = MagentoManager::runRequest(new Request("POST /Awardit_Altapay/callback/verify/quote/" . $quoteId, [
            "Content-Type" => "application/json",
        ], $body), 'default');

        $this->assertEquals((object)['result' => 'ok'], json_decode($response->getBody()));
        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));

        $quote->load($quoteId); // Reload from database
        // Order should have been created
        $this->assertNotEmpty($quote->getReservedOrderId());
        $order = Mage::getModel('sales/order')->loadByIncrementId($quote->getReservedOrderId());
        /** @var \Mage_Sales_Model_Order_Payment */
        $payment = $order->getPayment();
        $this->assertEquals('altapay', $payment->getMethod());
    }
}
