<?php

use Altapay\AbstractApi;
use Altapay\Api\Others\Payments;
use Altapay\Api\Payments\CaptureReservation;
use Altapay\Api\Payments\RefundCapturedReservation;
use Altapay\Api\Payments\ReleaseReservation;
use Altapay\Authentication;
use Altapay\Response\AbstractResponse;
use Altapay\Response\CaptureReservationResponse;
use Altapay\Response\Embeds\Transaction;
use Altapay\Response\RefundResponse;
use Altapay\Response\ReleaseReservationResponse;
use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\HandlerStack;
use Psr\Log\LoggerInterface;

class Awardit_Altapay_Model_Api_Merchant
{
    /** @var GuzzleClient */
    protected GuzzleClient $httpClient;

    /** @var ?Authentication */
    protected ?Authentication $authentication = null;

    public function __construct(?GuzzleClient $httpClient = null, ?LoggerInterface $logger = null)
    {
        if (null === $httpClient) {
            $handler = HandlerStack::create();
            if ($logger) {
                $logHandler = new Awardit_Altapay_Model_Api_LoggerMiddleware($logger);
                $handler->push($logHandler);
            }
            $httpClient = new GuzzleClient(['handler' => $handler]);
        }
        $this->httpClient = $httpClient;
    }

    /**
     * Altapay config
     * @return Authentication
     */
    public function getAuthentication(): Authentication
    {
        if (empty($this->authentication)) {
            $username = Mage::getStoreConfig('payment/altapay/username');
            $password = Mage::getStoreConfig('payment/altapay/password');
            $host = Mage::getStoreConfig('payment/altapay/host');
            $this->authentication = new Authentication($username, $password, $host);
        }
        return $this->authentication;
    }

    /**
     * Capture an authorized transaction
     * @param string $transactionId
     * @param float $amount
     * @return CaptureReservationResponse
     * @psalm-suppress InvalidReturnType, InvalidReturnStatement, PossiblyUnusedReturnValue
     */
    public function captureReservation(
        string $transactionId,
        float $amount
    ): CaptureReservationResponse {
        $api = new CaptureReservation($this->getAuthentication());
        $api->setTransaction($transactionId);
        $api->setAmount($amount);
        return $this->call($api);
    }

    /**
     * Undocumented function
     * @param string $transactionId
     * @return array<Transaction>
     * @psalm-suppress InvalidReturnType, InvalidReturnStatement, PossiblyUnusedReturnValue
     */
    public function getTransaction(string $transactionId): array
    {
        $api = new Payments($this->getAuthentication());
        $api->setTransaction($transactionId);
        return $this->call($api);
    }

    /**
     * Refund a captured transaction
     * @param string $transactionId
     * @param float $amount
     * @return RefundResponse
     * @psalm-suppress InvalidReturnType, InvalidReturnStatement, PossiblyUnusedReturnValue
     */
    public function refundCapturedReservation(
        string $transactionId,
        float $amount
    ): RefundResponse {
        $api = new RefundCapturedReservation($this->getAuthentication());
        $api->setTransaction($transactionId);
        $api->setAmount($amount);
        return $this->call($api);
    }

    /**
     * Release a reservation on an authorized transaction
     * @param string $transactionId
     * @return ReleaseReservationResponse
     * @psalm-suppress InvalidReturnType, InvalidReturnStatement, PossiblyUnusedReturnValue
     */
    public function releaseReservation(string $transactionId): ReleaseReservationResponse
    {
        $api = new ReleaseReservation($this->getAuthentication());
        $api->setTransaction($transactionId);
        return $this->call($api);
    }

    /**
     * Call api
     * @param AbstractApi $api
     * @return AbstractResponse|string|array<Transaction>
     * @psalm-suppress InvalidNullableReturnType
     */
    protected function call(AbstractApi $api)
    {
        try {
            $api->setClient($this->httpClient);
            return $api->call();
        } catch (\Throwable $e) {
            Mage::log("Error in Altapay Merchant API: {$e}");
            throw $e;
            // @todo do something
        }
    }
}
