<?php

namespace AltapayTests;

use AltapayTests\MockHelper;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use GuzzleHttp\Psr7\Response;
use Mage;
use Mage_Core_Model_Config;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;
use Varien_Object;

class GraphqlTest extends TestCase
{
    use MatchesSnapshots;

    public function setUp(): void
    {
        MagentoManager::reset();
        MagentoManager::init('default');

        $config = new Mage_Core_Model_Config();
        $config->saveConfig('payment/altapay/active', '1', 'default', 0);
        $config->saveConfig('payment/altapay/username', 'irrelevant', 'default', 0);
        $config->saveConfig('payment/altapay/password', 'irrelevant', 'default', 0);
        $config->saveConfig('payment/altapay/host', 'https://testgateway.altapaysecure.com', 'default', 0);
        Mage::app()->baseInit([]);
        Mage::getConfig()->loadModules()->loadDb()->saveCache(); // Reload config cache
    }

    public function testAltapaySetQuotePaymentType(): void
    {
        $result = $this->runQuery(
            '{ __type(name: "MutationSetQuotePaymentMethodAltapay") { name fields { name type { name } } } }'
        );
        $this->assertMatchesJsonSnapshot($result);
    }

    public function testAltapaySetQuotePayment(): void
    {
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        // Create a mock and queue two responses.
        $headers = ['Content-Type' => 'application/json'];
        $responses = [
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
        ];

        $requests = [];
        Mage::register('_helper/awardit_altapay', MockHelper::getHelper($responses, $requests));
        $result = $this->runQuery('mutation { setQuotePaymentMethod{ altapay { result sessionId token } } }');
        $this->assertMatchesJsonSnapshot($result);

        // Match that request sent is good
        $this->assertEquals(
            $requests[3]['request']->getUri()->getPath(),
            '/checkout/v1/api/session/542ef86d-af22-43f6-abd8-b23c84b77f9e'
        );
        $body = preg_replace([
            '|"orderId":"[0-9]+"|',
            '|"quoteId":"[0-9]+"|',
            '|([a-z]+)_gallery|',
        ], [
            '"orderId":"<ORDERID>"',
            '"quoteId":"<QUOTEID>"',
            '<GALLERY>_gallery'
        ], $requests[3]['request']->getBody());
        $this->assertMatchesJsonSnapshot($body);
    }

    /* --------- Helper methods ------------------------------------------------------ */

    private function runQuery(string $query): object
    {
        $response = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], $query), 'default');
        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        return json_decode($response->getBody());
    }
}
