<?php

namespace AltapayTests;

use AltapayTests\MockHelper;
use Awardit_Altapay_Model_Method_Altapay;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use GuzzleHttp\Psr7\Response;
use Mage;
use Mage_Core_Model_Config;
use Mage_Sales_Model_Quote;
use MageQL_Sales_Helper_Quote;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;
use Varien_Object;

class GraphqlTest extends TestCase
{
    use MatchesSnapshots;

    public function setUp(): void
    {
        MagentoManager::reset();
        MagentoManager::init('default');

        $config = new Mage_Core_Model_Config();
        $config->saveConfig('payment/altapay/active', '1', 'default', 0);
        $config->saveConfig('payment/altapay/username', 'irrelevant', 'default', 0);
        $config->saveConfig('payment/altapay/password', 'irrelevant', 'default', 0);
        $config->saveConfig('payment/altapay/host', 'https://testgateway.altapaysecure.com', 'default', 0);
        Mage::app()->baseInit([]);
        Mage::getConfig()->loadModules()->loadDb()->saveCache(); // Reload config cache
    }

    public function testAltapaySetQuotePaymentType(): void
    {
        $result = $this->runQuery(
            '{ __type(name: "MutationSetQuotePaymentMethodAltapay") { name fields { name type { name } } } }'
        );
        $this->assertMatchesJsonSnapshot($result);
    }

    private function quote(): Mage_Sales_Model_Quote
    {
        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel('catalog/product')
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel('catalog/product')->getIdBySku('test-simple'));

        // Plain quantity request
        $request = new Varien_Object([
            'qty' => 1,
        ]);
        $quote = Mage::getSingleton('checkout/session')->getQuote();
        $quote->setCustomerEmail('example@email.com');

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel('sales/quote_address')->addData([
            'postcode' => '12345',
            'lastname' => 'Testsson',
            'street' => 'Paradisäppelvägen 111',
            'city' => 'Ankeborg',
            'email' => 'test@example.com',
            'telephone' => '0123456789',
            'country_id' => 'SE',
            'firstname' => 'Test',
        ]));
        $quote->setShippingAddress(Mage::getModel('sales/quote_address')->addData([
            'postcode' => '12345',
            'lastname' => 'Testsson',
            'street' => 'Paradisäppelvägen 111',
            'city' => 'Ankeborg',
            'email' => 'test@example.com',
            'telephone' => '0123456789',
            'country_id' => 'SE',
            'firstname' => 'Test',
        ]));
        $quote->addProduct($product, $request);
        $quote->setIsActive(1);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setShippingMethod('flatrate_flatrate');
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();
        return $quote;
    }

    public function testAltapaySetQuotePayment(): void
    {
        MagentoManager::init();

        // Create a mock and queue two responses.
        $headers = ['Content-Type' => 'application/json'];
        $responses = [
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
        ];

        $requests = [];
        Mage::register('_helper/awardit_altapay', MockHelper::getHelper($responses, $requests));

        $this->quote();

        $result = $this->runQuery('mutation { setQuotePaymentMethod{ altapay { result sessionId token } } }');
        $this->assertMatchesJsonSnapshot($result);

        // Match that request sent is good
        $this->assertEquals(
            $requests[3]['request']->getUri()->getPath(),
            '/checkout/v1/api/session/542ef86d-af22-43f6-abd8-b23c84b77f9e'
        );
        $body = preg_replace([
            '|"orderId":"[0-9]+"|',
            '|"quoteId":"[0-9]+"|',
            '|([a-z]+)_gallery|',
        ], [
            '"orderId":"<ORDERID>"',
            '"quoteId":"<QUOTEID>"',
            '<GALLERY>_gallery'
        ], $requests[3]['request']->getBody());
        $this->assertMatchesJsonSnapshot($body);
    }

    public function testAltapayGetTokenFromSessionId(): void
    {
        MagentoManager::init();

        // Create a mock and queue two responses.
        $headers = ['Content-Type' => 'application/json'];
        $responses = [
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
        ];
         $requests = [];
        Mage::register('_helper/awardit_altapay', MockHelper::getHelper($responses, $requests));

        $result = $this->runQuery('query { checkoutToken(sessionId: "542ef86d-af22-43f6-abd8-b23c84b77f9e") }');
        $this->assertMatchesJsonSnapshot($result);
    }

    public function testAltapayGetTokenFromQuote(): void
    {
        MagentoManager::init();
         // Create a mock and queue two responses.
        $headers = ['Content-Type' => 'application/json'];
        $responses = [
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
        ];
         $requests = [];
        Mage::register('_helper/awardit_altapay', MockHelper::getHelper($responses, $requests));

        $quote = $this->quote();
        $model = Mage::getSingleton('mageql_sales/quote');
        $payment = $model->importPaymentData($quote, 'altapay', []);
        /** @var Awardit_Altapay_Model_Method_Altapay */
        $method = $payment->getMethodInstance();
        $method->selectSession($quote);
        $model->saveSessionQuote();

        $result = $this->runQuery('query { checkoutToken }');
        $this->assertMatchesJsonSnapshot($result);
    }

    public function testAltapayGetTokenFromOrder(): void
    {
        MagentoManager::init();
         // Create a mock and queue two responses.
        $headers = ['Content-Type' => 'application/json'];
        $responses = [
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/token.json')),
            new Response(200, $headers, file_get_contents(__DIR__ . '/fixtures/sessionId.json')),
        ];
         $requests = [];
        Mage::register('_helper/awardit_altapay', MockHelper::getHelper($responses, $requests));

        $quote = $this->quote();
        $model = Mage::getSingleton('mageql_sales/quote');
        $payment = $model->importPaymentData($quote, 'altapay', []);
        /** @var Awardit_Altapay_Model_Method_Altapay */
        $method = $payment->getMethodInstance();
        $method->selectSession($quote);
        $model->saveSessionQuote();

        $session = Mage::getSingleton('checkout/session');
        $helper = Mage::helper('mageql_sales/quote');
        $order = $helper->submitOrder($quote);

        $session->unsetAll();
        $session->clearHelperData();
        $session->setLastQuoteId((int)$quote->getId());
        $session->setLastSuccessQuoteId((int)$quote->getId());
        $session->setLastOrderId((int)$order->getId());
        $session->setLastRealOrderId($order->getIncrementId());

        $result = $this->runQuery('query { checkoutToken }');
        $this->assertEmpty($session->getQuote()->getPayment()->getAdditionalInformation('checkoutToken'));
        $this->assertMatchesJsonSnapshot($result);
    }

    /* --------- Helper methods ------------------------------------------------------ */

    private function runQuery(string $query): object
    {
        $response = MagentoManager::runRequest(new Request('POST /graphql', [
            'Content-Type' => 'application/graphql',
        ], $query), 'default');
        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals('application/json; charset=utf-8', $response->getHeader('Content-Type'));
        return json_decode($response->getBody());
    }
}
