<?php

namespace AltapayTests;

use Awardit_Altapay_Model_Method_Altapay;
use Awardit_Altapay_Model_Type_NotificationCallback;
use Crossroads\Magento\Test\Integration\MagentoManager;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Mage;
use Mage_Core_Model_Config;
use Mage_Core_Model_Store;
use Mage_Sales_Model_Order_Payment;
use Mage_Sales_Model_Quote;
use PHPUnit\Framework\TestCase;
use Varien_Object;

class CheckoutTest extends TestCase
{
    private Mage_Core_Model_Store $store;

    public function setUp(): void
    {
        MagentoManager::reset();
        MagentoManager::init('default');

        $config = new Mage_Core_Model_Config();
        $config->saveConfig('payment/altapay/active', '1', 'default', 0);
        $config->saveConfig('payment/altapay/payment_action', 'authorize', 'default', 0);
        $config->saveConfig('payment/altapay/username', 'irrelevant', 'default', 0);
        $config->saveConfig('payment/altapay/password', 'irrelevant', 'default', 0);
        $config->saveConfig('payment/altapay/host', 'https://testgateway.altapaysecure.com', 'default', 0);
        $config->saveConfig('payment/altapay/cb_verify', 'https://test.verify', 'default', 0);
        $config->saveConfig('payment/altapay/cb_notification', 'https://test.notification', 'default', 0);
        $config->saveConfig('payment/altapay/shopName', 'test-shop', 'default', 0);
        Mage::app()->baseInit([]);
        Mage::getConfig()->loadModules()->loadDb()->saveCache(); // Reload config cache

        $this->store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
    }

    private function quoteCreate(): Mage_Sales_Model_Quote
    {

        $quote = Mage::getModel('sales/quote')->setStoreId($this->store->getId());
        $quote->getBillingAddress();
        $quote->getShippingAddress();
        $quote->setIsActive(1);
        $quote->save();
        return $quote;
    }

    private function quoteAddProduct(Mage_Sales_Model_Quote $quote, string $sku, float $qty = 1): Mage_Sales_Model_Quote
    {
        $product = Mage::getModel('catalog/product')
            ->setStoreId($this->store->getId())
            ->load(Mage::getModel('catalog/product')->getIdBySku($sku));
        $quote->addProduct($product, new Varien_Object(['qty' => $qty]));
        $quote->collectTotals();
        $quote->save();
        return $quote;
    }

    private function quoteSetBillingAddress(Mage_Sales_Model_Quote $quote, array $address = []): Mage_Sales_Model_Quote
    {
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData($address));
        $quote->save();
        return $quote;
    }

    private function quoteSetShippingAddress(Mage_Sales_Model_Quote $quote, array $address = []): Mage_Sales_Model_Quote
    {
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData($address));
        $quote->save();
        return $quote;
    }

    private function quoteSetShippingMethod(Mage_Sales_Model_Quote $quote, string $method): Mage_Sales_Model_Quote
    {
        $quote->getShippingAddress()->setShippingMethod($method);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();
        return $quote;
    }

    private function quoteSetPaymentMethod(Mage_Sales_Model_Quote $quote, string $method): Mage_Sales_Model_Quote
    {
        $quote->getPayment()->setMethod($method);
        $quote->save();
        return $quote;
    }

    public function testPhysicalSuccess(): void
    {
        // Create a mock and queue two responses.
        $requests = [];
        $responses = [
            $this->createResponse('checkout.authenticate.merchant.success'),
            $this->createResponse('checkout.session.create.success'),
            $this->createResponse('checkout.authenticate.customer.success'),
            $this->createResponse('checkout.session.update.success'),
            $this->createResponse('checkout.session.read.success'),
        ];
        Mage::register('_helper/awardit_altapay', MockHelper::getHelper($responses, $requests));

        // Set up quote with physical product and address
        $address = [
            'firstname'  => 'Jane',
            'lastname'   => 'Doe',
            'street'     => 'Alfagatan 1',
            'city'       => 'Mölndal',
            'postcode'   => '43147',
            'country_id' => 'SE',
            'telephone'  => '07001234567',
        ];
        $quote = $this->quoteCreate();
        $this->quoteAddProduct($quote, 'test-simple');
        $this->quoteSetBillingAddress($quote, $address);
        $this->quoteSetShippingAddress($quote, $address);
        $this->quoteSetShippingMethod($quote, 'flatrate_flatrate');
        $this->quoteSetPaymentMethod($quote, 'altapay');

        $payment = $quote->getPayment();
        $this->assertEquals('altapay', $payment->getMethod());
        $this->assertEquals(25.71, $quote->getGrandTotal());
        $this->assertEquals('SEK', $quote->getQuoteCurrencyCode());
        $this->assertEquals(1, $quote->getIsActive());
        $this->assertEmpty($payment->getAdditionalInformation('sessionId'));
        $this->assertEmpty($payment->getAdditionalInformation('checkoutToken'));

        // Initiate altapay checkout session
        $altapay = new Awardit_Altapay_Model_Method_Altapay();
        $altapay->selectSession($quote);

        // Check what we send
        $this->assertCount(4, $requests);
        $this->assertRequest('checkout.authenticate.merchant.success', $requests[0]['request']);
        $this->assertRequest('checkout.session.create.success', $requests[1]['request']);
        $this->assertRequest('checkout.authenticate.customer.success', $requests[2]['request']);
        $this->assertRequest('checkout.session.update.success', $requests[3]['request'], [
            '<ORDERID>' => $quote->getReservedOrderId(),
            '<QUOTEID>' => $quote->getId(),
        ]);
        // Check stored identifiers
        $this->assertEquals('<SESSIONID>', $payment->getAdditionalInformation('sessionId'));
        $this->assertEquals('<CUSTOMERTOKEN>', $payment->getAdditionalInformation('checkoutToken'));

        // Checkout order, as if Altapay has called verify callback
        $verifyCb = new Awardit_Altapay_Model_Type_NotificationCallback([
            'order' => [
                'orderId' => $quote->getReservedOrderId(),
                'amount' => 25.71,
                'currency' => 'SEK',
            ],
            'status' => 'NEW',
            'requiresCapture' => true,
            'method' => [

            ],
            'payment' => [
                'gateway' => [
                    'publicPaymentId' => '<PUBLICPAYMENTID>',
                    'paymentId' => '<PAYMENTID>',
                ],
            ],
        ]);
        $altapay->createOrder($quote, $verifyCb);

        // Check what we send
        $this->assertCount(5, $requests);
        $this->assertRequest('checkout.session.read.success', $requests[4]['request']);
        $this->assertRequest('checkout.session.update.success', $requests[3]['request'], [
            '<ORDERID>' => $quote->getReservedOrderId(),
            '<QUOTEID>' => $quote->getId(),
        ]);

        // Should now have an order in review and more identifiers stored
        $order = Mage::getModel('sales/order')->loadByIncrementId($quote->getReservedOrderId());
        /** @var Mage_Sales_Model_Order_Payment */
        $payment = $order->getPayment();
        $this->assertEquals($quote->getReservedOrderId(), $order->getIncrementId());
        $this->assertEquals('payment_review', $order->getStatus());
        $this->assertEquals('payment_review', $order->getState());
        $this->assertEquals(0, $quote->getIsActive());
        $this->assertEquals('<SESSIONID>', $payment->getAdditionalInformation('sessionId'));
        $this->assertEquals('<CUSTOMERTOKEN>', $payment->getAdditionalInformation('checkoutToken'));
        $this->assertEquals(true, $payment->getAdditionalInformation('requiresCapture'));
        $this->assertEquals('NEW', $payment->getAdditionalInformation('status'));
        $this->assertEquals('<PUBLICPAYMENTID>', $payment->getAdditionalInformation('payment.gateway.publicPaymentId'));
        $this->assertEquals('<PAYMENTID>', $payment->getAdditionalInformation('payment.gateway.paymentId'));
        $this->assertNotEmpty($payment->getAuthorizationTransaction());

        // Finilize order, as if Altapay has called notification callback
        $verifyCb = new Awardit_Altapay_Model_Type_NotificationCallback([
            'order' => [
                'orderId' => $order->getIncrementId(),
                'amount' => 25.71,
                'currency' => 'SEK',
            ],
            'status' => 'SUCCEEDED',
            'requiresCapture' => true,
            'method' => [
            ],
            'payment' => [
                'gateway' => [
                    'publicPaymentId' => '<PUBLICPAYMENTID>',
                    'paymentId' => '<PAYMENTID>',
                ],
            ],
        ]);
        $altapay->updateOrder($order, $verifyCb);

        // Should now be processing
        $this->assertEquals('processing', $order->getStatus());
        $this->assertEquals('processing', $order->getState());
        $this->assertEquals('SUCCEEDED', $payment->getAdditionalInformation('status'));
    }

    private function createResponse(string $source): Response
    {
        $source = json_decode(file_get_contents(__DIR__ . "/fixtures/{$source}.response.json"));
        $headers = array_merge([
            'Content-Type' => 'application/json',
        ], (array)$source->headers);
        return new Response($source->code, $headers, json_encode($source->body));
    }

    private function assertRequest(string $expected, Request $request, array $replacers = []): void
    {
        $expected = file_get_contents(__DIR__ . "/fixtures/{$expected}.request.json");
        foreach ($replacers as $target => $replacement) {
            $expected = preg_replace("|{$target}|", $replacement, $expected);
        }
        $expected = json_decode($expected);
        $this->assertEquals($expected->method, $request->getMethod());
        $this->assertEquals($expected->uri, (string)$request->getUri());
        foreach ($expected->headers as $name => $content) {
            $this->assertEquals($content, $request->getHeaderLine($name));
        }
        $this->assertEquals($expected->body, json_decode($request->getBody()->__toString()));
    }
}
