<?php

use Psr\Log\LoggerInterface;

class Awardit_Altapay_Model_Observer
{
    private static array $capture_on = [
        Mage_Sales_Model_Order::STATE_PROCESSING,
    ];
    private static array $complete_on = [
        Mage_Sales_Model_Order::STATE_PROCESSING,
        Mage_Sales_Model_Order::STATE_COMPLETE,
    ];

    /** @var Awardit_Altapay_Helper_Data */
    private $helper;
    private LoggerInterface $logger;

    public function __construct()
    {
        $this->helper = Mage::helper('awardit_altapay');
        $this->logger = $this->helper->getPsrLogger();
    }

    /**
     * If Quote is changed and assoicated with Altapay, update Altapay session.
     */
    public function onQuoteSave(Varien_Event_Observer $observer): void
    {
        $quote = $observer->getQuote();
        $payment = $quote->getPayment();
        if ($payment->getMethod() != 'altapay') {
            return;
        }
        $method = $payment->getMethodInstance();
        $method->updateSession($quote);
    }

    /**
     * If Order state is changed and assoicated with Altapay, possibly take actions
     */
    public function onOrderCommit(Varien_Event_Observer $observer): void
    {
        $order = $observer->getOrder();
        $payment = $order->getPayment();
        if ($payment->getMethod() != 'altapay') {
            return; // Not eligable
        }
        if ($order->getState() == $order->getOrigData('state')) {
            return; // No state change
        }
        $logContext = [
            'orderId' => $order->getIncrementId(),
            'oldState' => $order->getOrigData('state'),
            'newState' => $order->getState(),
        ];
        $this->logger->debug('Order {orderId} changed state {oldState} > {newState}', $logContext);

        // Possibly capture
        if (
            $order->getIsVirtual()
            && in_array($order->getState(), self::$capture_on)
            && $payment->canCapture()
            && $payment->getAdditionalInformation('requiresCapture')
        ) {
            // Capture directly if only virtual items
            try {
                $payment->capture(null);
                $payment->save();
                $this->logger->debug('Capture on {orderId} (virtual order)', $logContext);
            } catch (Exception $e) {
                $logContext['exception'] = $e;
                $this->logger->error('Failed capture on {orderId}', $logContext);
            }
        }

        // Possibly send codes/mark as completed
        if (in_array($order->getState(), self::$complete_on)) {
            // Send any virtual products since we have an authorization approved
            Mage::dispatchEvent('crossroads_order_payment_complete', ['order' => $order]);
            $this->logger->debug('Dispatch crossroads_order_payment_complete on {orderId}', $logContext);
        }
    }

    public function onMageqlContextCreateAfter(): void
    {
        $session = Mage::getSingleton('checkout/session');
        $quoteId = $session->getQuoteId();
        if (!$quoteId) {
            return;
        }
        $quote = Mage::getModel('sales/quote')->setStoreId(Mage::app()->getStore()->getId())->load($quoteId);
        if (!$quote->getId() || $quote->getPayment()->getMethod() != 'altapay') {
            return;
        }

        if ($quote->getIsActive()) {
            // Ensure quote in session
            $session->setQuoteId((int)$quote->getId());
            if ($session->getLastSuccessQuoteId() == $quote->getId()) {
                $session->setLastSuccessQuoteId(null);
            }
            return;
        }

        if (!$quote->getReservedOrderId()) {
            return;
        }

        $order = Mage::getModel('sales/order')->loadByIncrementId($quote->getReservedOrderId());
        if (!$order->getId()) {
            return;
        }

        $session->clear();
        if ($order->isCanceled()) {
            $session->setLastSuccessQuoteId(null);
        } else {
            $session->setLastSuccessQuoteId((int)$quote->getId());
        }
        $session->setLastOrderId((int)$order->getId());
        $session->setLastRealOrderId($order->getIncrementId());
    }
}
