<?php

use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Normalizer\ArrayDenormalizer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;

class Awardit_Altapay_CallbackController extends Mage_Core_Controller_Front_Action
{
    protected ?Serializer $serializer = null;

    /**
     * Get serializer
     * @return Serializer
     */
    protected function getSerializer(): Serializer
    {
        if (empty($this->serializer)) {
            $normalizers = [
                new ArrayDenormalizer(),
                new ObjectNormalizer(),
            ];
            $this->serializer = new Serializer($normalizers, [new JsonEncoder()]);
        }
        return $this->serializer;
    }

    /**
     * @param int $code
     * @param mixed $body
     * @return Zend_Controller_Response_Abstract
     * @throws Zend_Controller_Response_Exception
     */
    protected function jsonResponse($code, $body)
    {
        return $this->getResponse()
            ->setHttpResponseCode($code)
            ->setHeader('Content-Type', 'application/json; charset=utf-8', true)
            ->setBody(json_encode($body, Mage::getIsDeveloperMode() ? JSON_PRETTY_PRINT : 0));
    }

    /**
     * Failure callback from Altapay
     * @return Zend_Controller_Response_Abstract
     */
    public function failureAction()
    {
        return $this->jsonResponse(200, [
            'result' => 'ok',
        ]);
    }

    /**
     * Notification callback from Altapay
     * @return Zend_Controller_Response_Abstract
     */
    public function notificationAction()
    {
        $helper = Mage::helper('awardit_altapay');
        $req = $this->getRequest();

        if ('POST' !== $req->getMethod()) {
            return $this->jsonResponse(405, [
                'message' => 'Method not allowed',
            ]);
        }

        try {
            $body = $this->getSerializer()->deserialize(
                $req->getRawBody(),
                'Awardit_Altapay_Model_Type_NotificationCallback',
                'json'
            );
        } catch (Exception $e) {
            $helper->log(sprintf(
                '%s %s malformed request: %s',
                __METHOD__,
                $e->getMessage(),
                (string)$req->getRawBody()
            ));
            return $this->jsonResponse(400, [
                'message' => 'Malformed request',
            ]);
        }

        if (empty($body) || empty($body->order) || empty($body->order['orderId'])) {
            $helper->log(sprintf('%s malformed request: %s', __METHOD__, (string)$req->getRawBody()));
            return $this->jsonResponse(400, [
                'message' => 'Malformed request',
            ]);
        }

        /** @var Mage_Sales_Model_Order */
        $order = Mage::getModel('sales/order')->loadByIncrementId($body->order['orderId']);

        if (!$order->getId()) {
            $helper->log(sprintf(
                '%s order id not found: %s %s',
                __METHOD__,
                $body->order['orderId'],
                (string)$req->getRawBody()
            ));
            return $this->jsonResponse(404, [
                'message' => 'Order id not found',
            ]);
        }

        $helper->log(sprintf(
            '%s updating order %s (orderId: %s), request: %s',
            __METHOD__,
            $order->getIncrementId(),
            $order->getId(),
            (string)$req->getRawBody()
        ));
        /** @var Mage_Sales_Model_Order_Payment */
        $payment = $order->getPayment();
        /** @var Awardit_Altapay_Model_Method_Altapay */
        $method = $payment->getMethodInstance();

        // Update Magento order
        try {
            $method->updateOrder($order, $body);
            return $this->jsonResponse(200, [
                'result' => 'ok',
            ]);
        } catch (Awardit_Altapay_Exception $e) {
            return $this->jsonResponse($e->getCode(), [
                'message' => $e->getHttpMessage(),
            ]);
        } catch (Throwable $e) {
            return $this->jsonResponse(500, [
                'message' => 'Server error',
            ]);
        }
    }

    /**
     * Verify callback from Altapay
     * @return Zend_Controller_Response_Abstract
     */
    public function verifyAction()
    {
        $req = $this->getRequest();
        $helper = Mage::helper('awardit_altapay');

        if ('POST' !== $req->getMethod()) {
            return $this->jsonResponse(405, [
                'message' => 'Method not allowed',
                'errorMessage' => $helper->__('Technical error. Please contact support.'),
            ]);
        }

        try {
            $body = $this->getSerializer()->deserialize(
                $req->getRawBody(),
                'Awardit_Altapay_Model_Type_NotificationCallback',
                'json'
            );
        } catch (Exception $e) {
            $helper->log(sprintf(
                '%s %s malformed request: %s',
                __METHOD__,
                $e->getMessage(),
                (string)$req->getRawBody()
            ));
            return $this->jsonResponse(400, [
                'message' => 'Malformed request',
                'errorMessage' => $helper->__('Technical error. Please contact support.'),
            ]);
        }

        if (empty($body) || empty($body->order) || empty($body->order['orderId'])) {
            $helper->log(sprintf('%s malformed request: %s', __METHOD__, (string)$req->getRawBody()));
            return $this->jsonResponse(400, [
                'message' => 'Malformed request',
                'errorMessage' => $helper->__('Technical error. Please contact support.'),
            ]);
        }

        $quote = $helper->getQuoteByOrderId($body->order['orderId']);

        if (!$quote) {
            $helper->log(sprintf(
                '%s quote id not found: %s %s',
                __METHOD__,
                $body->order['orderId'],
                (string)$req->getRawBody()
            ));
            return $this->jsonResponse(404, [
                'message' => 'Quote id not found',
                'errorMessage' => $helper->__('Technical error. Please contact support.'),
            ]);
        }

        $helper->log(sprintf(
            '%s verifying quote %s, request: %s',
            __METHOD__,
            $quote->getId(),
            (string)$req->getRawBody()
        ));
        $payment = $quote->getPayment();
        /** @var Awardit_Altapay_Model_Method_Altapay */
        $method = $payment->getMethodInstance();

        // Create Magento order
        try {
            $method->createOrder($quote, $body);
            return $this->jsonResponse(200, [
                'result' => 'ok',
            ]);
        } catch (Awardit_Altapay_Exception $e) {
            $helper->log(sprintf(
                '%s could not create order from quote %s, error: %s',
                __METHOD__,
                $quote->getId(),
                $e->getHttpMessage()
            ));
            return $this->jsonResponse($e->getCode(), [
                'message' => $e->getHttpMessage(),
                'errorMessage' => $helper->__('An error occured. Please try again.'),
            ]);
        } catch (Throwable $e) {
            $helper->log(sprintf(
                '%s could not create order from quote %s, error: %s',
                __METHOD__,
                $quote->getId(),
                $e->getMessage()
            ));
            return $this->jsonResponse(500, [
                'message' => 'Server error',
                'errorMessage' => $helper->__('Technical error. Please contact support.'),
            ]);
        }
    }

    /**
     * Success callback from Altapay
     * @return Zend_Controller_Response_Abstract
     */
    public function successAction()
    {
        return $this->jsonResponse(200, [
            'result' => 'ok',
        ]);
    }
}
