<?php

use Awardit_Altapay_Model_Checkout_Type_Session as Session;

class Awardit_Altapay_Model_QuoteHandler
{
    private Awardit_Altapay_Helper_Data $helper;
    private Mage_Sales_Model_Quote $quote;
    private Mage_Sales_Model_Quote_Payment $payment;

    public function __construct(Mage_Sales_Model_Quote $quote)
    {
        $this->helper = Mage::helper('awardit_altapay');
        $this->quote = $quote;
        $this->payment = $quote->getPayment();
    }


    /* ---------- Session handling --------------------------------------------------- */

    public function hasSession(): bool
    {
        return $this->getInformationNullableString('sessionId') !== null;
    }

    public function getSessionId(): string
    {
        return $this->getInformationString('sessionId');
    }

    public function getSession(): Session
    {
        if (!$this->hasSession()) {
            throw new Awardit_Altapay_Exception("No session on {$this->getOrderId()}");
        }
        // Will throw exception if missing or expired
        $session = $this->helper->checkout()->getSession($this->getInformationString('sessionId'));
        $this->setInformation($session->getInformation());
        return $session;
    }

    /** @api */
    public function createSession(Session $create): Session
    {
        $session = $this->helper->checkout()->createSession($create);
        $this->setInformation([
            'sessionId' => $session->sessionId,
            'checksum' => $this->calculateChecksum($create),
        ]);
        $this->setInformation($session->getInformation());
        return $session;
    }

    /** @api */
    public function updateSession(Session $update): Session
    {
        $session = $this->helper->checkout()->updateSession($this->getInformationString('sessionId'), $update);
        if ($session === null) {
            $session = $this->helper->checkout()->getSession($this->getInformationString('sessionId'));
        }
        $this->setInformation([
            'checksum' => $this->calculateChecksum($update),
        ]);
        $this->setInformation($session->getInformation());
        return $session;
    }

    /** @api */
    public function resetSession(): void
    {
        $this->setInformation(['sessionId' => null]);
    }


    /* ---------- Checkout token handling -------------------------------------------- */

    /** @api */
    public function createCheckoutToken(): string
    {
        $token = $this->helper->customerCheckoutToken($this->getInformationString('sessionId'))->token;
        $this->setInformation(['checkoutToken' => $token]);
        return $token;
    }


    /* ---------- Checksum handling -------------------------------------------------- */

    public function hasChecksumChanged(Session $session): bool
    {
        return $this->calculateChecksum($session) !== $this->getInformationString('checksum');
    }


    /* ---------- Payment handling --------------------------------------------------- */

    public function hasPayment(): bool
    {
        return $this->getInformationNullableString('session.activePayment.externalPaymentId') !== null;
    }

    public function getPaymentId(): string
    {
        return $this->getInformationString('session.activePayment.externalPaymentId');
    }


    /* ---------- Quote management --------------------------------------------------- */

    public function reset(): void
    {
        $this->setInformation([
            'sessionId' => null,
            'checkoutToken' => null,
            'checksum' => null,
        ]);
        $this->quote->setReservedOrderId('');
    }

    public function isActive(): bool
    {
        return (bool)$this->quote->getIsActive();
    }

    public function isAltapay(): bool
    {
        return $this->payment->getMethod() === 'altapay';
    }

    public function getOrderId(): string
    {
        $this->quote->reserveOrderId(); // Create if missing
        return $this->quote->getReservedOrderId();
    }

    public function setInformation(array $information): void
    {
        foreach ($information as $key => $value) {
            $this->payment->setAdditionalInformation($key, $value);
        }
        $this->payment->save();
    }


    /* ---------- Internal helpers -------------------------------------------------- */

    private function getInformationString(string $key): string
    {
        return (string)$this->getInformationNullableString($key);
    }

    private function getInformationNullableString(string $key): ?string
    {
        /** @var string|null $value */
        $value = $this->payment->getAdditionalInformation($key);
        return $value === null || $value === '' ? null : $value;
    }

    private function calculateChecksum(Session $session): string
    {
        return md5(json_encode($session->order));
    }
}
