<?php

use Awardit_Altapay_Model_Checkout_Type_Session as Session;
use GraphQL\Error\ClientAware;
use Psr\Log\LoggerInterface;

/**
 * @psalm-type AltapayTransaction = object{}
 * Altapay class is wrongly defined
 */
class Awardit_Altapay_Model_Method_Altapay extends Mage_Payment_Model_Method_Abstract
{
    /** @var mixed */
    protected $_code = 'altapay';
    /** @var bool */
    protected $_isGateway = true;
    /** @var mixed */
    protected $_canAuthorize = true;
    /** @var mixed */
    protected $_canCapture = true;
    /** @var mixed */
    protected $_canRefund = true;
    /** @var mixed */
    protected $_canRefundInvoicePartial = true;
    /** @var mixed */
    protected $_canVoid = true;
    /** @var mixed */
    protected $_canUseInternal = false;
    /** @var mixed */
    protected $_canUseCheckout = true;
    /** @var mixed */
    protected $_canUseForMultishipping = false;
    /** @var mixed */
    protected $_canFetchTransactionInfo = false;
    /** @var mixed */
    protected $_canManageRecurringProfiles = false;

    /** @var array */
    protected $_dirty = [];

    /** @var Awardit_Altapay_Helper_Data */
    protected $helper;
    protected LoggerInterface $logger;

    /**
     * Constructor for class.
     */
    public function __construct()
    {
        $this->helper = Mage::helper('awardit_altapay');
        $this->logger = $this->helper->getPsrLogger();
    }

    /**
     * Check whether payment method can be used.
     * @param Mage_Sales_Model_Quote|null $quote
     * @return bool
     */
    public function isAvailable($quote = null)
    {
        if (!parent::isAvailable($quote)) {
            return false;
        }
        /** @psalm-suppress RiskyTruthyFalsyComparison */
        if (empty($quote) || empty($quote->getId()) || empty($quote->getItemsCount())) {
            return false;
        }
        if (!$this->canUseForCountry($quote->getBillingAddress()->getCountry())) {
            return false;
        }
        if (!$this->canUseForCurrency($quote->getQuoteCurrencyCode())) {
            return false;
        }
        if (!$this->canUseCheckout()) {
            return false;
        }
        $total = $quote->getBaseSubtotal() + $quote->getShippingAddress()->getBaseShippingAmount();
        if ($total < 0.0001) {
            return false;
        }
        return true;
    }

    /**
     * Configure session for checkout (stores sessionId for future use) when method is selected.
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @return $this
     * @psalm-suppress UnusedVariable, TypeDoesNotContainType, NoValue, RedundantCondition, InvalidCast
     */
    public function selectSession(Mage_Sales_Model_Quote $quote): self
    {
        $quote->reserveOrderId();
        $quoteHandler = new Awardit_Altapay_Model_QuoteHandler($quote);

        $logContext = [
            'action' => 'session.select',
            'orderId' => $quoteHandler->getOrderId(),
            'quoteId' => $quote->getId(),
            'sessionId' => $quoteHandler->getSessionId(),
            'storeCode' => $quote->getStore()->getCode(),
        ];

        // Check if existing session still available
        if ($quoteHandler->hasSession()) {
            try {
                $session = $quoteHandler->getSession();
                $this->logger->debug('{orderId} {action} | Read session {sessionId}', $logContext);
            } catch (Awardit_Altapay_Model_Checkout_Exception_NotFoundException $e) {
                $quoteHandler->resetSession();
                $this->logger->warning("{orderId} {action} | Expired session {sessionId}", $logContext);
            }
        }

        // Create new checkout session
        if (!$quoteHandler->hasSession()) {
            $create = $this->getCheckoutConfiguration($quote->getStore());
            $session = $quoteHandler->createSession($create);
            $logContext['sessionId'] = $session->sessionId;
            $this->logger->debug('{orderId} {action} | Created session {sessionId}', $logContext);

            // Additional metadata
            $quoteHandler->setInformation([
                'checksum' => null,
                'method' => $this->getCode(),
            ]);
        }

        // Generate new token
        $quoteHandler->createCheckoutToken();

        $this->updateSession($quote);

        return $this;
    }

    /**
     * Updates session based on quote. Triggered by observer when quote is changed.
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @return $this
     */
    public function updateSession(Mage_Sales_Model_Quote $quote): self
    {
        $quoteHandler = new Awardit_Altapay_Model_QuoteHandler($quote);
        if (!$quoteHandler->isActive() || !$quoteHandler->isAltapay() || !$quoteHandler->hasSession()) {
            return $this; // No update of closed quotes
        }

        $original = $quoteHandler->getOrderId();
        $logContext = [
            'action' => 'session.update',
            'orderId' => $original,
            'quoteId' => $quote->getId(),
            'sessionId' => $quoteHandler->getSessionId(),
            'storeCode' => $quote->getStore()->getCode(),
        ];

        $quote->reserveOrderId(); // Ensure unique order increment
        if ($quoteHandler->getOrderId() != $original) {
            $logContext['original'] = $original;
            $logContext['orderId'] = $quoteHandler->getOrderId();
            $this->logger->debug("{orderId} {action} | Updated reserved id {original} to {orderId}", $logContext);
        }

        // Check if Altapay needs to be updated
        $update = $this->getCheckoutSessionFromQuote($quote);
        if (!$quoteHandler->hasChecksumChanged($update)) {
            $this->logger->debug("{orderId} {action} | Session {sessionId} need no update", $logContext);
            return $this;
        }

        try {
            // Update Altapay with Quote data
            $quoteHandler->updateSession($update);
            $this->logger->debug("{orderId} {action} | Updated session {sessionId}", $logContext);
        } catch (Awardit_Altapay_Model_Checkout_Exception_BadRequestException $e) {
            // Set a temporary attribute that is picked up later by
            // Awardit_Altapay_Model_Observer::onCustomerAddressValidationAfter
            $quote->setData('altapay_error_code', $e->getErrorResponse()->errorCode);
        }
        return $this;
    }

    /**
     * Create a Checkout Session configuration.
     * @return Session Session configuration
     */
    private function getCheckoutConfiguration(Mage_Core_Model_Store $store): Session
    {
        // @todo: Enable more fields as needed
        $data = $this->filter([
            'callbacks' => $this->filter([
                'success' => ['type' => 'FUNCTION'],
                'failure' => ['type' => 'FUNCTION'],
                // 'redirect' => '',
                'notification' => (string)$store->getConfig('payment/altapay/cb_notification'),
                'verifyOrder' => (string)$store->getConfig('payment/altapay/cb_verify'),
                // 'formStyling' => '',
                'bodyFormat' => 'JSON',
            ]),
            'configuration' => $this->filter([
                'paymentType' => 'PAYMENT',
                'paymentDisplayType' => 'SCRIPT',
                // 'agreement' => [],
                'autoCapture' => $store->getConfig('payment/altapay/payment_action') == 'authorize_capture',
                // Must match Altapay config, contact them
                'shopName' => (string)$store->getConfig('payment/altapay/shop_name'),
                // Default country, address country override when set
                'country' => Mage::helper('core')->getDefaultCountry($store),
                'language' => substr($store->getConfig('general/locale/code') ?? 'en', 0, 2),
                // 'ccToken' => [],
            ]),
        ]);
        return new Session($data);
    }

    /**
     * Create a Checkout Session from Quote data.
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @return Session
     */
    private function getCheckoutSessionFromQuote(Mage_Sales_Model_Quote $quote): Session
    {
        $data = ['order' => $this->filter([
            'orderId' => (string)$quote->getReservedOrderId(),
            'amount' => [
                'value' =>  (float)$quote->getGrandTotal(),
                'currency' => $quote->getQuoteCurrencyCode(),
            ],
            'orderLines' => $this->getOrderLines($quote),
            'customer' => $this->getCustomer($quote),
            'transactionInfo' => [
                'quoteId' => $quote->getId(),
                'storeCode' => $quote->getStore()->getCode(),
                'sessionToken' => $this->helper->getSessionTokenFromQuote($quote),
            ],
        ])];
        return new Session($data);
    }

    /**
     * Get shipping fee item from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress MismatchingDocblockParamType
     */
    private function getShippingItem(Mage_Core_Model_Abstract $source): array
    {
        $source = $source instanceof Mage_Sales_Model_Quote ? $source->getShippingAddress() : $source;
        $unitPrice = round($source->getShippingAmount(), 2);
        return $this->filter($unitPrice == 0 ? [] : [
            'goodsType' => 'shipment',
            'itemId' => 'shipment',
            'description' => $source->getShippingDescription(),
            'quantity' => 1,
            'unitPrice' => $unitPrice,
            'taxAmount' => round($source->getShippingTaxAmount() + $source->getShippingHiddenTaxAmount(), 2),
            'discountPercent' => 0, // @todo: Not sure about the discount stuff
        ]);
    }

    /**
     * Get payment fee item from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress MismatchingDocblockParamType
     */
    private function getPaymentFeeItem(Mage_Core_Model_Abstract $source): array
    {
        if ($source instanceof Mage_Sales_Model_Quote) {
            $totals = $source->getTotals();
            if (!isset($totals['payment_fee'])) {
                return [];
            }
            $fee = $totals['payment_fee'];
            $excl = round($fee->getValueExclTax(), 2);
            $tax = round($fee->getValueInclTax() - $fee->getValueExclTax(), 2);
            $name = $fee->getTitle();
        } else {
            $excl = round($source->getPaymentFeeAmount(), 2);
            $tax = round($source->getPaymentFeeTaxAmount(), 2);
            $name = $source->getPaymentFeeTitle();
        }
        return $excl > 0 ? $this->filter([
            'goodsType' => 'handling',
            'itemId' => 'payment',
            'description' => $name ?: 'Payment fee',
            'quantity' => 1,
            'unitPrice' => $excl,
            'taxAmount' => $tax,
            'discountPercent' => 0, // @todo: Not sure about the discount stuff
        ]) : [];
    }

    /**
     * Get customer info from Order or Quote.
     * Source priority: 1) Order|Quote 2) Billing address 3) Shipping address
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress RedundantConditionGivenDocblockType
     */
    private function getCustomer(Mage_Core_Model_Abstract $source): array
    {
        $customer = [];
        if ($shippingAddress = $source->getShippingAddress()) {
            $customer = array_merge($customer, $this->filter([
                'firstName' => $shippingAddress->getFirstname(),
                'lastName' => $shippingAddress->getLastname(),
                'email' => $shippingAddress->getEmail(),
                'phoneNumber' => $shippingAddress->getTelephone(),
                'shippingAddress' => $shippingAddress->validate() === true ? $this->filter([
                    'firstName' => $shippingAddress->getFirstname(),
                    'lastName' => $shippingAddress->getLastname(),
                    'street' => $shippingAddress->getStreetFull(),
                    'city' => $shippingAddress->getCity(),
                    'region' => $shippingAddress->getRegion(),
                    'country' => $shippingAddress->getCountry(),
                    'zipCode' => $shippingAddress->getPostcode(),
                ]) : null,
            ]));
        }
        if ($billingAddress = $source->getBillingAddress()) {
            $customer = array_merge($customer, $this->filter([
                'firstName' => $billingAddress->getFirstname(),
                'lastName' => $billingAddress->getLastname(),
                'email' => $billingAddress->getEmail(),
                'phoneNumber' => $billingAddress->getTelephone(),
                'billingAddress' => $billingAddress->validate() === true ? $this->filter([
                    'firstName' => $billingAddress->getFirstname(),
                    'lastName' => $billingAddress->getLastname(),
                    'street' => $billingAddress->getStreetFull(),
                    'city' => $billingAddress->getCity(),
                    'region' => $billingAddress->getRegion(),
                    'country' => $billingAddress->getCountry(),
                    'zipCode' => $billingAddress->getPostcode(),
                ]) : null,
            ]));
        }
        $customer = array_merge($customer, $this->filter([
            'firstName' => $source->getCustomerFirstname(),
            'lastName' => $source->getCustomerLastname(),
            'email' => $source->getCustomerEmail(),
        ]));
        return $this->filter($customer);
    }

    /**
     * Get order items from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress MismatchingDocblockParamType, UndefinedMagicMethod
     */
    private function getOrderItems(Mage_Core_Model_Abstract $source): array
    {
        $isOrder = $source instanceof Mage_Sales_Model_Order;
        return array_map(function ($item) use ($source, $isOrder) {
            $product = $item->getProduct();
            $product->setStoreId($source->getStoreId())->load((int) $product->getId());
            $type = $item->getIsVirtual() ? 'digital' : 'item';
            $images = $product->getMediaGalleryImages();
            if (!$isOrder) {
                $item->calcTaxAmount();
            }
            return $this->filter([
                'itemId' => $product->getSku(),
                'description' => $product->getName(),
                'quantity' => $isOrder ? $item->getQtyOrdered() : $item->getQty(),
                'unitPrice' => $isOrder ? $item->getPrice() : $item->getConvertedPrice(),
                'taxAmount' =>  $isOrder ? $item->getTaxAmount() : $item->getTaxBeforeDiscount(),
                'taxPercent' => (float)$item->getTaxPercent(),
                'discountPercent' => (float)$item->getDiscountPercent(),
                'goodsType' => $type,
                'imageUrl' => $images->getSize() > 0 ? $images->getFirstItem()->getUrl() : null,
                'productUrl' => $product->getUrlInStore(),
            ]);
        }, $source->getAllVisibleItems());
    }

    /**
     * Get discount items from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     */
    private function getDiscountItem(Mage_Core_Model_Abstract $source): array
    {
        if ($source instanceof Mage_Sales_Model_Quote) {
            $totals = $source->getTotals();
            if (!isset($totals['discount']) || $totals["discount"]->getValue() === 0) {
                return [];
            }
            $amount = $totals["discount"]->getValue();
        } else {
            $amount = $source->getDiscountAmount();
        }
        $amount = round($amount, 2);

        return $amount == 0 ? [] : $this->filter([
            'goodsType' => 'discount',
            'itemId' => 'discount',
            'description' => 'Discount',
            'quantity' => 1,
            'unitPrice' => $amount,
            'taxAmount' => 0, // @todo: Not sure what to do here
            'discountPercent' => 0, // @todo: Not sure about the discount stuff
        ]);
    }

    /**
     * Get Retain24 discount items from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     */
    private function getRetain24Item(Mage_Core_Model_Abstract $source): array
    {
        if ($source instanceof Mage_Sales_Model_Quote) {
            $totals = $source->getTotals();
            if (!isset($totals['retain24']) || $totals["retain24"]->getValue() === 0) {
                return [];
            }
            $amount = $totals["retain24"]->getValue();
        } else {
            $amount = $source->getRetain24Amount();
        }
        $amount = round($amount, 2);

        return $amount == 0 ? [] : $this->filter([
            'goodsType' => 'discount',
            'itemId' => 'discount',
            'description' => 'Discount',
            'quantity' => 1,
            'unitPrice' => $amount,
            'taxAmount' => 0, // @todo: Not sure what to do here
            'discountPercent' => 0, // @todo: Not sure about the discount stuff
        ]);
    }

    /**
     * Get all order lines (including fees) from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress MismatchingDocblockParamType
     */
    private function getOrderLines(Mage_Core_Model_Abstract $source): array
    {
        return array_values($this->filter(array_merge($this->getOrderItems($source), [
            $this->getShippingItem($source),
            $this->getPaymentFeeItem($source),
            $this->getDiscountItem($source),
            $this->getRetain24Item($source),
        ])));
    }

    /**
     * Create Magento Order, update Altapay session accordingly.
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @return Mage_Sales_Model_Order
     * @throws Awardit_Altapay_Exception if unable to create order and/or update session
     * @psalm-suppress PossiblyUnusedReturnValue, InvalidReturnType
     */
    public function createOrder(
        Mage_Sales_Model_Quote $quote,
        Awardit_Altapay_Model_Type_NotificationCallback $notification
    ): Mage_Sales_Model_Order {
        $quoteHandler = new Awardit_Altapay_Model_QuoteHandler($quote);
        if (!$quoteHandler->isActive()) {
            $this->helper->throw("Quote {$quote->getId()} not active", 400);
        }
        if (!$quoteHandler->isAltapay()) {
            $this->helper->throw("Quote {$quote->getId()} with wrong payment method", 400);
        }
        if (!$quoteHandler->hasSession()) {
            $this->helper->throw("Quote {$quote->getId()} have no session", 400);
        }

        $session = $quoteHandler->getSession(); // Throws error if missing

        // @todo: temporary fallback, remove when stable
        if ($quoteHandler->hasPayment()) {
            $paymentId = $quoteHandler->getPaymentId();
        } else {
            $this->logger->debug('{orderId} {action} | No payment.id on session');
            $paymentId = $notification->payment['gateway']['paymentId'];
        }

        $logContext = [
            'action' => 'order.create',
            'orderId' => $quoteHandler->getOrderId(),
            'paymentId' => $paymentId,
            'quoteId' => $quote->getId(),
            'sessionId' => $session->sessionId,
            'status' => $notification->status,
            'storeCode' => $quote->getStore()->getCode(),
        ];

        // Check if Quote has invalid changes
        $update = $this->getCheckoutSessionFromQuote($quote);
        if ($quoteHandler->hasChecksumChanged($update)) {
            $this->logger->error('{orderId} {action} | Invalid changes on quote', $logContext);
            $this->helper->throw("Invalid changes on quote/{$quote->getId()}");
        }

        $this->logger->debug('{orderId} {action} | Creating order', $logContext);

        // Set dummy address
        $store = $quote->getStore();

        /** @var Mage_Sales_Model_Quote_Address */
        $billingAddress = $quote->getBillingAddress();
        if (
            $billingAddress->validate() !== true &&
            $quote->isVirtual() &&
            Mage::helper("mageql_sales")->getAllowPlaceholderBillingAddress($store)
        ) {
            $defaultCountryId = $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY) ?? '';
            $billingAddress
                ->setAddressType(Mage_Sales_Model_Quote_Address::TYPE_BILLING)
                ->setFirstname($billingAddress->getFirstname() ?: "-")
                ->setLastname($billingAddress->getLastname() ?: "-")
                ->setStreet($billingAddress->getStreet(-1) ?: ["-"])
                ->setPostcode($billingAddress->getPostcode() ?: "-")
                ->setTelephone($billingAddress->getTelephone() ?: "-")
                ->setCity($billingAddress->getCity() ?: "-")
                ->setCountryId($billingAddress->getCountryId() ?: $defaultCountryId);
        }

        // Finalize Quote
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        try {
            // Create Magento Order
            $service = Mage::getModel('sales/service_quote', $quote);
            $service->submitAll();

            $order = $service->getOrder();
            $payment = $order->getPayment();

            $this->logger->info('{orderId} {action} | Created order', $logContext);

            // Close Quote
            $quote->setIsActive(0);
            $quote->save();

            if ($payment) {
                $this->addNotificationToPayment($payment, $notification);
                $payment->save();
            }

            return $order;
        } catch (Mage_Core_Exception $e) {
            $logContext['exception'] = $e;
            $logContext['error'] = $e->getMessage();
            $this->logger->error('{orderId} {action} | Failed creating order: {error}', $logContext);
            if ($e instanceof ClientAware) {
                throw $e; // Already user friendly
            }
            $this->helper->throw("Failed creating order for quote/{$quote->getId()}: {$e->getMessage()}");
        }
    }

    private function checkPayment(Mage_Sales_Model_Order_Payment $payment, Session $session): void
    {
        if (empty($session->activePayment)) {
            return;
        }
        $this->addNotificationToPaymentRecursive($payment, [
            'paymentId' => $session->activePayment['paymentId'] ?? null,
            'externalPaymentId' => $session->activePayment['externalPaymentId'] ?? null,
            'status' => $session->activePayment['status'] ?? null,
        ], 'session.activePayment.');
        if (empty($session->activePayment['externalPaymentId'])) {
            return;
        }
        $transactions = $this->helper->merchant()->getTransactions($this->getPaymentId($payment));
        if (count($transactions) == 0) {
            $this->logger->warning("No transaction on order {orderId} payment {paymentId}", [
                'ordeId' => $session->order['orderId'],
                'paymentId' => $this->getPaymentId($payment),
            ]);
            return;
        }
        if (count($transactions) > 1) {
            $this->logger->warning("{count} transactions on order {orderId} payment {paymentId}, investigate", [
                'ordeId' => $session->order['orderId'],
                'paymentId' => $this->getPaymentId($payment),
                'count' => count($transactions),
            ]);
        }
        /** @var AltapayTransaction $transaction */
        $transaction = $transactions[0];
        $cardInformation = $transaction->CardInformation;
        $this->addNotificationToPaymentRecursive($payment, [
            'payment' => [
                'id' => $transaction->PaymentId,
                'type' => $transaction->PaymentNature,
                'scheme' => $transaction->PaymentSchemeName,
            ],
            'transaction' => [
                'id' => $transaction->TransactionId,
                'status' => $transaction->TransactionStatus,
                'reason' => $transaction->ReasonCode,
            ],
            'card' => [
                'status' => $transaction->CardStatus,
                'pan' => $cardInformation->MaskedPan,
                'expiry' => "{$cardInformation->Expiry->Year}-{$cardInformation->Expiry->Month}",
                'scheme' => "{$cardInformation->Scheme} {$cardInformation->IssuingCountry}",
            ],
            'shop' => $transaction->Shop,
            'shopOrderId' => $transaction->ShopOrderId,
            'amount' => [
                'reserved' => $transaction->ReservedAmount,
                'captured' => $transaction->CapturedAmount,
                'refunded' => $transaction->RefundedAmount,
                'surcharge' => $transaction->SurchargeAmount,
            ],
            'date' => [
                'created' => $transaction->CreatedDate->format('c'),
                'updated' => $transaction->UpdatedDate->format('c'),
            ],
            'supports' => [
                'refunds' => $transaction->PaymentNatureService->SupportsRefunds ? '1' : '0',
                'release' => $transaction->PaymentNatureService->SupportsRelease ? '1' : '0',
                'multipleCaptures' => $transaction->PaymentNatureService->SupportsMultipleCaptures ? '1' : '0',
                'multipleRefunds' => $transaction->PaymentNatureService->SupportsMultipleRefunds ? '1' : '0',
            ],
            'fraud' => [
                'riskScore' => $transaction->FraudRiskScore,
                'explanation' => $transaction->FraudExplanation,
            ],
        ], 'payment.transaction.');
    }

    private function addNotificationToPayment(
        Mage_Sales_Model_Order_Payment $payment,
        Awardit_Altapay_Model_Type_NotificationCallback $notification
    ): void {
        $this->addNotificationToPaymentRecursive($payment, [
            'requiresCapture' => $notification->requiresCapture,
            'status' => $notification->status,
            'sessionId' => $notification->sessionId,
            'cardInformation' => $notification->cardInformation,
            'method' => $notification->method,
            'payment' => $notification->payment,
            'riskAnalysis' => $notification->riskAnalysis,
        ]);
    }

    private function addNotificationToPaymentRecursive(
        Mage_Sales_Model_Order_Payment $payment,
        array $data,
        string $prefix = ''
    ): void {
        foreach ($data as $key => $value) {
            if (empty($value)) {
                continue;
            } elseif (is_array($value)) {
                $this->addNotificationToPaymentRecursive($payment, $value, "{$prefix}{$key}.");
            } elseif (is_scalar($value)) {
                $payment->setAdditionalInformation("{$prefix}{$key}", $value);
                $payment->setTransactionAdditionalInfo("{$prefix}{$key}", (string)$value);
            }
        }
        $combined = array_filter(array_merge(
            $payment->getAdditionalInformation() ?? [], // payment info
            $payment->getTransactionAdditionalInfo() ?? [] // current transaction info
        ), function ($item) {
            return !empty($item) && is_scalar($item);
        });
        ksort($combined);
        /** @psalm-suppress InvalidCast, InvalidArgument */
        $payment->setTransactionAdditionalInfo(
            Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS,
            $combined
        );
    }

    /**
     * Handle a webhhook "notification" request from Altapay.
     * @psalm-suppress PossiblyUnusedReturnValue
     */
    public function updateOrder(
        Mage_Sales_Model_Order $order,
        Awardit_Altapay_Model_Type_NotificationCallback $notification
    ): void {
        /** @var Mage_Sales_Model_Order_Payment */
        $payment = $order->getPayment();
        $client = $this->helper->checkout();
        /** @var string */
        $sessionId = $payment->getAdditionalInformation('sessionId');
        $session = $client->getSession($sessionId);
        // @todo Remove fallback
        $status = $session->activePayment['status'] ?? $notification->status;
        $incrementId = $order->getIncrementId();
        $paymentId = $notification->payment['gateway']['paymentId'];
        $logContext = [
            'action' => 'order.update',
            'orderId' => $incrementId,
            'paymentId' => $paymentId,
            'sessionId' => $sessionId,
            'status' => $status,
            'storeCode' => $order->getStore()->getCode(),
        ];
        $this->logger->debug("{orderId} {action} | status: {status}", $logContext);

        // @todo Just checking as of now - this is where we get externalPaymentId
        $this->checkPayment($payment, $session);

        switch (strtolower($status)) {
            case 'succeeded':
                $transaction = $payment->getAuthorizationTransaction();
                if (empty($transaction)) {
                    return; // Not able to handle?
                }
                $logContext['txn'] = $transaction->getTxnId();
                if ($transaction->getIsClosed()) {
                    return; // Already handled
                }
                $payment->setTransactionId($transaction->getTxnId());
                // @todo Session has priority
                $this->addNotificationToPayment($payment, $notification);
                $payment->setNotificationResult(true);
                $payment->authorize(false, $notification->order['amount']);
                $order->setCanSendNewEmailFlag(true);
                $this->logger->debug("{orderId} {action} | authorized {txn}", $logContext);

                try {
                    $order->queueNewOrderEmail();
                } catch (Throwable $e) {
                    $logContext['exception'] = $e;
                    $logContext['message'] = $e->getMessage();
                     $this->logger->error('{orderId} {action} | Exception sending email: {message}', $logContext);
                }

                // For some reason we need a "order" transaction to capture from admin
                $orderTransaction = $payment->lookupTransaction(
                    "{$transaction->getTxnId()}-order",
                    Mage_Sales_Model_Order_Payment_Transaction::TYPE_ORDER
                );
                if (empty($orderTransaction)) {
                    Mage::getModel('sales/order_payment_transaction')
                        ->setTxnId("{$transaction->getTxnId()}-order")
                        ->setOrderPaymentObject($payment)
                        ->setTxnType(Mage_Sales_Model_Order_Payment_Transaction::TYPE_ORDER)
                        ->setParentTxnId($transaction->getTxnId())
                        ->save();
                }

                break;
            case 'cancelled':
            case 'declined':
            case 'error':
            case 'failed':
                $logContext['error'] = $notification->merchantErrorMessage;

                // Checkout failed or cancelled by customer after initiated payment
                if (!$order->canCancel()) {
                    $this->logger->warning("{orderId} {action} | Can not cancel order", $logContext);
                    return;
                }
                // Cancel order
                // @todo Session has priority
                $this->addNotificationToPayment($payment, $notification);
                $payment->setIsTransactionClosed(true);
                $payment->save();
                $order->cancel("{$notification->merchantErrorMessage} ({$notification->status})");
                $order->save();
                $this->logger->debug("{orderId} {action} | Cancelled order", $logContext);

                // Re-open quote, if still exist
                $quoteId = $order->getQuoteId();
                $quote = Mage::getModel('sales/quote')->load($quoteId);
                if ((int) $quote->getId() === 0) {
                    break;
                }

                // Reset quote payment session
                $quotePayment = $quote->getPayment();
                $quotePayment->setAdditionalInformation('sessionId', null);
                $quotePayment->setAdditionalInformation('checkoutToken', null);
                $quotePayment->save();

                // Re-open quote with new order number
                $quote->setIsActive(1);
                $quote->setReservedOrderId(''); // Reset reserved
                $quote->save();
                $logContext['quoteId'] = $quote->getId();
                $this->logger->debug("{orderId} {action} | Re-opened quote {quoteId}.", $logContext);
                break;
            case 'unknown':
                // @todo: What to do on these?
                $payment->setIsTransactionDenied(true);
                $payment->setIsTransactionClosed(true);
                break;
            case 'new':
            case 'pending':
                // @todo: not sure what to do here
                $payment->setIsTransactionPending(true);
                break;
            default:
                $this->helper->throw("Unhandled Altapay status: {$notification->status}");
        }

        $payment->save();
        $order->save();
    }

    /**
     * Handle a webhook "notification" request from Altapay.
     */
    public function updateQuote(
        Mage_Sales_Model_Quote $quote,
        Awardit_Altapay_Model_Type_NotificationCallback $notification
    ): void {
        $quoteHandler = new Awardit_Altapay_Model_QuoteHandler($quote);

        $reservedId = $quote->getReservedOrderId();
        $paymentId = $notification->payment['gateway']['paymentId'];
        $logContext = [
            'action' => 'quote.update',
            'orderId' => $reservedId,
            'paymentId' => $paymentId,
            'quoteId' => $quote->getId(),
            'sessionId' => $notification->sessionId,
            'status' => $notification->status,
            'storeCode' => $quote->getStore()->getCode(),
        ];
        $this->logger->debug("{orderId} {action} | status: {status}", $logContext);

        /** @psalm-suppress RedundantFunctionCallGivenDocblockType */
        switch (strtolower($notification->status)) {
            case 'succeeded':
                // @todo Fallback order creation to be added here
                $this->logger->warning("{orderId} {action} | Should not be succeeded", $logContext);
                break;
            case 'cancelled':
            case 'declined':
            case 'error':
            case 'failed':
                $quoteHandler->reset();
                $logContext['error'] = $notification->merchantErrorMessage;
                $this->logger->warning("{orderId} {action} | Failed update with {status}, reset", $logContext);
                break;
            case 'unknown':
            case 'new':
            case 'pending':
                // @todo: not sure what to do here
                $this->logger->warning("{orderId} {action} | Unnown status {status}, ignore", $logContext);
                break;
            default:
                $this->helper->throw("Unhandled Altapay status: {$notification->status}");
        }

        $quote->save();
    }

    /**
     * Authorize payment method.
     * @param Mage_Sales_Model_Order_Payment $payment
     * @param float $amount
     * @return $this
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function authorize(Varien_Object $payment, $amount): self
    {
        parent::authorize($payment, $amount); // Checks availability

        $order = $this->getOrder($payment);
        /** @var string */
        $sessionId = $payment->getAdditionalInformation('sessionId');
        $logContext = [
            'action' => 'payment.authorize',
            'amount' => $amount,
            'orderId' => $order->getIncrementId(),
            'sessionId' => $sessionId,
            'storeCode' => $order->getStore()->getCode(),
        ];
        $this->logger->debug("{orderId} {action} | Authorization", $logContext);

        // @todo: Verify Altapay?

        $payment->setTransactionId($sessionId);
        $payment->setIsTransactionDenied(false);
        $payment->setIsTransactionClosed(false);
        $payment->setIsTransactionPending(true);
        $payment->setIsTransactionApproved(true);
        $payment->save();

        return $this;
    }

    /**
     * Capture payment, either directly or on previously authorized payment.
     * @param Mage_Sales_Model_Order_Payment $payment
     * @param float $amount
     * @return $this
     * @throws Awardit_Altapay_Exception on failure
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function capture(Varien_Object $payment, $amount): self
    {
        parent::capture($payment, $amount); // Checks availability

        $merchant = $this->helper->merchant();
        $order = $this->getOrder($payment);
        /** @var string */
        $sessionId = $payment->getAdditionalInformation('sessionId');
        $transactionId = $this->getPaymentId($payment);
        $logContext = [
            'action' => 'payment.capture',
            'orderId' => $order->getIncrementId(),
            'paymentId' => $transactionId,
            'sessionId' => $sessionId,
            'storeCode' => $order->getStore()->getCode(),
            'amount' => $amount,
        ];

        $payment->setTransactionId("{$sessionId}-capture");
        try {
            $merchant->captureReservation($transactionId, $amount); // @todo save additional info?
            $order->setCanSendNewEmailFlag(true);
            $this->logger->info("{orderId} {action} | Capture complete", $logContext);
        } catch (Throwable $e) {
            $logContext['exception'] = $e;
            $this->logger->error("{orderId} {action} | Capture failed", $logContext);
            $this->helper->throw("Altapay capture failed: {$e->getMessage()}.");
        }

        return $this;
    }

    /**
     * Check with Altapay transaction data if we can cancel payment
     *
     * @param Mage_Sales_Model_Order_Payment $payment
     * @return boolean
     * @psalm-suppress MoreSpecificImplementedParamType
     */
    public function canVoid(Varien_Object $payment)
    {
        $canVoid = parent::canVoid($payment);
        if (!$canVoid) {
            return $canVoid;
        }

        return (bool) $payment->getAdditionalInformation('payment.transaction.supports.release');
    }

    /**
     * Cancel authorization.
     * @param Mage_Sales_Model_Order_Payment $payment
     * @return $this
     * @throws Awardit_Altapay_Exception on failure
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function cancel(Varien_Object $payment): self
    {
        parent::cancel($payment); // Checks availability

        $payment->setIsTransactionDenied(true);
        $payment->setIsTransactionClosed(true);

        $merchant = $this->helper->merchant();
        $order = $this->getOrder($payment);
        $transactionId = $this->getPaymentId($payment);
        $logContext = [
            'action' => 'payment.cancel',
            'orderId' => $order->getIncrementId(),
            'paymentId' => $transactionId,
            'sessionId' => $payment->getAdditionalInformation('sessionId'),
            'storeCode' => $order->getStore()->getCode(),
        ];

        try {
            $merchant->releaseReservation($transactionId); // @todo save additional info?
            $this->logger->info("{orderId} {action} | Cancel complete ", $logContext);
        } catch (Throwable $e) {
            $logContext['exception'] = $e;
            $this->logger->error("{orderId} {action} | Cancel failed", $logContext);
            $this->helper->throw("Altapay cancel failed: {$e->getMessage()}.");
        }

        $payment->save();
        return $this;
    }

    /**
     * Void, same as cancel.
     * @param  Mage_Sales_Model_Order_Payment $payment
     * @return $this
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function void(Varien_Object $payment): self
    {
        parent::void($payment); // Checks availability
        return $this->cancel($payment);
    }

    /**
     * Check with Altapay transaction data if we can refund payment
     *
     * @return boolean
     */
    public function canRefund()
    {
        $canRefund = parent::canRefund();
        if (!$canRefund) {
            return $canRefund;
        }

        $order = $this->getInfoInstance()->getOrder();
        $payment = $order ? $order->getPayment() : null;

        return $payment ? (bool) $payment->getAdditionalInformation('payment.transaction.supports.refunds') : false;
    }

    /**
     * Refund specified amount for payment
     * @param Mage_Sales_Model_Order_Payment $payment
     * @param float $amount
     * @return $this
     * @throws Awardit_Altapay_Exception on failure
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function refund(Varien_Object $payment, $amount): self
    {
        parent::refund($payment, $amount); // Checks availability

        $payment->setIsTransactionDenied(true);
        $payment->setIsTransactionClosed(true);

        $merchant = $this->helper->merchant();
        $order = $this->getOrder($payment);
        $transactionId = $this->getPaymentId($payment);
        $logContext = [
            'action' => 'payment.refund',
            'orderId' => $order->getIncrementId(),
            'paymentId' => $transactionId,
            'sessionId' => $payment->getAdditionalInformation('sessionId'),
            'storeCode' => $order->getStore()->getCode(),
        ];

        try {
            $merchant->refundCapturedReservation($transactionId, $amount); // @todo save additional info?
            $this->logger->info("{orderId} {action} | Refund complete", $logContext);
        } catch (Throwable $e) {
            $logContext['exception'] = $e;
            $this->logger->error("{orderId} {action} | Refund failed", $logContext);
            $this->helper->throw("Altapay refund failed: {$e->getMessage()}.");
        }

        $payment->save();
        return $this;
    }

    /**
     * Get Checkout paymentId (used as Merchant transactionId) from payment
     * @throws Awardit_Altapay_Exception if paymentId can not be found
     * @psalm-suppress InvalidReturnType, InvalidReturnStatement, RiskyTruthyFalsyComparison
     */
    private function getPaymentId(Mage_Sales_Model_Order_Payment $payment): string
    {
        // From session, secure
        if ($paymentId = $payment->getAdditionalInformation('session.activePayment.externalPaymentId')) {
            return $paymentId;
        }
        // From webhook, insecure @todo remove
        if ($paymentId = $payment->getAdditionalInformation('payment.gateway.paymentId')) {
            return $paymentId;
        }
        $this->logger->error("{orderId} {action} | Could not resolve paymentId", [
            'action' => 'session.paymentId',
            'orderId' => $this->getOrder($payment)->getIncrementId(),
        ]);
        $this->helper->throw("Could not resolve Altapay paymentId.");
    }

    /**
     * Sometimes Order is not pre-handled on Payment. This method ensures it is.
     * @param Mage_Sales_Model_Order_Payment $payment Magento payment
     * @return Mage_Sales_Model_Order
     */
    private function getOrder(Mage_Sales_Model_Order_Payment $payment): Mage_Sales_Model_Order
    {
        $order = $payment->getOrder();
        if (empty($order)) {
            $order = Mage::getModel('sales/order');
            $order->load($payment->getParentId());
            $payment->setOrder($order);
        }
        return $order;
    }

    // Don't filter booleans
    private function filter(array $source): array
    {
        return array_filter($source, function ($item) {
            return is_bool($item) || !empty($item);
        });
    }
}
