<?php

use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Request;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Normalizer\ArrayDenormalizer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;
use Symfony\Component\Serializer\SerializerInterface;

final class Awardit_Altapay_Model_Checkout_Client
{
    /** @var Awardit_Altapay_Model_Checkout_Configuration */
    protected $config;

    /** @var ClientInterface */
    protected $httpClient;

    /** @var SerializerInterface */
    protected $serializer;

    public function __construct(
        Awardit_Altapay_Model_Checkout_Configuration $config,
        ClientInterface $httpClient,
        SerializerInterface $serializer
    ) {
        $this->config = $config;
        $this->httpClient = $httpClient;
        $this->serializer = $serializer;
    }

    /**
    * @param null|Awardit_Altapay_Model_Checkout_Type_AuthenticateRequest $requestBody
    * @throws Awardit_Altapay_Model_Checkout_Exception_BadRequestException
    * @throws Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException
    *
    * @return Awardit_Altapay_Model_Checkout_Type_TokenResponse
    */
    public function authenticate(?Awardit_Altapay_Model_Checkout_Type_AuthenticateRequest $requestBody = null)
    {
        return $this->executeEndpoint(
            new Awardit_Altapay_Model_Checkout_Endpoint_Authentication($this->config, $requestBody)
        );
    }

    /**
    * @param null|Awardit_Altapay_Model_Checkout_Type_Session $requestBody
    * @throws Awardit_Altapay_Model_Checkout_Exception_BadRequestException
    * @throws Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException
    *
    * @return Awardit_Altapay_Model_Checkout_Type_Session
    */
    public function createSession(
        ?Awardit_Altapay_Model_Checkout_Type_Session $requestBody = null
    ): Awardit_Altapay_Model_Checkout_Type_Session {
        return $this->executeEndpoint(
            new Awardit_Altapay_Model_Checkout_Endpoint_CreateSession($this->config, $requestBody)
        );
    }

    /**
    * @param string $sessionId Unique session ID associated with user's checkout session
    * @throws Awardit_Altapay_Model_Checkout_Exception_NotFoundException
    * @throws Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException
    *
    * @return Awardit_Altapay_Model_Checkout_Type_Session
    */
    public function getSession($sessionId): Awardit_Altapay_Model_Checkout_Type_Session
    {
        return $this->executeEndpoint(
            new Awardit_Altapay_Model_Checkout_Endpoint_GetSession($this->config, $sessionId)
        );
    }

    /**
    * @param string $sessionId Unique session ID associated with user's checkout session
    * @param null|Awardit_Altapay_Model_Checkout_Type_Session $requestBody
    * @throws Awardit_Altapay_Model_Checkout_Exception_BadRequestException
    * @throws Awardit_Altapay_Model_Checkout_Exception_NotFoundException
    * @throws Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException
    *
    * @return Awardit_Altapay_Model_Checkout_Type_Session|null
    * @psalm-suppress PossiblyUnusedReturnValue
    */
    public function updateSession($sessionId, ?Awardit_Altapay_Model_Checkout_Type_Session $requestBody = null)
    {
        return $this->executeEndpoint(
            new Awardit_Altapay_Model_Checkout_Endpoint_UpdateSession($this->config, $sessionId, $requestBody)
        );
    }

    /**
    * @param null|Awardit_Altapay_Model_Checkout_Type_CreatePaymentRequest $requestBody
    * @throws Awardit_Altapay_Model_Checkout_Exception_BadRequestException
    * @throws Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException
    *
    * @return Awardit_Altapay_Model_Checkout_Type_Payment|ArrayObject
    * @psalm-suppress PossiblyUnusedReturnValue
    */
    public function createPayment(?Awardit_Altapay_Model_Checkout_Type_CreatePaymentRequest $requestBody = null)
    {
        return $this->executeEndpoint(
            new Awardit_Altapay_Model_Checkout_Endpoint_CreatePayment($this->config, $requestBody)
        );
    }

    /**
    * @param string $paymentId Unique payment ID
    * @throws Awardit_Altapay_Model_Checkout_Exception_NotFoundException
    * @throws Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException
    *
    * @return Awardit_Altapay_Model_Checkout_Type_Payment|ArrayObject
    */
    public function getPayment($paymentId)
    {
        return $this->executeEndpoint(
            new Awardit_Altapay_Model_Checkout_Endpoint_GetPayment($this->config, $paymentId)
        );
    }

    /**
    * @param string $sessionId Unique session ID associated with user's checkout session
    * @throws Awardit_Altapay_Model_Checkout_Exception_NotFoundException
    * @throws Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException
    *
    * @return Awardit_Altapay_Model_Checkout_Type_PaymentMethodResponse|ArrayObject
    */
    public function getPaymentMethods($sessionId)
    {
        return $this->executeEndpoint(
            new Awardit_Altapay_Model_Checkout_Endpoint_GetPaymentMethods($this->config, $sessionId)
        );
    }


    /**
    * @param string $sessionId Unique session ID associated with user's checkout session
    * @throws Awardit_Altapay_Model_Checkout_Exception_NotFoundException
    * @throws Awardit_Altapay_Model_Checkout_Exception_UnauthorizedException
    *
    * @return Awardit_Altapay_Model_Checkout_Type_Payment|ArrayObject
    */
    public function getSessionPayments($sessionId)
    {
        return $this->executeEndpoint(
            new Awardit_Altapay_Model_Checkout_Endpoint_GetSessionPayments($this->config, $sessionId)
        );
    }

    /**
     * @psalm-suppress MissingReturnType
     */
    protected function executeEndpoint(Awardit_Altapay_Model_Checkout_Endpoint_Interface $endpoint)
    {
        return $endpoint->parseResponse($this->processEndpoint($endpoint), $this->serializer);
    }

    private function processEndpoint(Awardit_Altapay_Model_Checkout_Endpoint_Interface $endpoint): ResponseInterface
    {
        $queryString = $endpoint->getQueryString();
        $uriGlue = false === strpos($endpoint->getUri(), '?') ? '?' : '&';
        $uri = $queryString !== '' ? $endpoint->getUri() . $uriGlue . $queryString : $endpoint->getUri();
        $method = $endpoint->getMethod();
        $headers = $endpoint->getHeaders();
        $body = $endpoint->getBody($this->serializer);
        $request = new Request($method, $uri, $headers, $body);
        return $this->httpClient->send($request, ['http_errors' => false]);
    }

    public static function create(
        Awardit_Altapay_Model_Checkout_Configuration $config,
        ?Client $httpClient = null,
        ?LoggerInterface $logger = null
        // array $additionalPlugins = [],
        // array $additionalNormalizers = []
    ): self {

        if (null === $httpClient) {
            $handler = HandlerStack::create();
            if ($logger) {
                $logHandler = new Awardit_Altapay_Model_Api_LoggerMiddleware($logger);
                $handler->push($logHandler);
            }
            $httpClient = new Client([
                'base_uri' => $config->getHost(),
                'handler' => $handler,
                'headers' => [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                    'User-Agent' => 'awardit/altapay',
                ],
            ]);
        }
        $normalizers = [
            new ArrayDenormalizer(),
            new Awardit_Altapay_Model_Checkout_Normalizer_PaymentMethodResponseNormalizer(),
            new ObjectNormalizer(),
        ];
        $serializer = new Serializer($normalizers, [new JsonEncoder()]);
        return new static($config, $httpClient, $serializer);
    }
}
