<?php

use GraphQL\Type\Definition\ResolveInfo;
use MageQL\Registry;

class Awardit_Altapay_Model_Graphql_Payment extends MageQL_Sales_Model_Payment_Abstract
{
    /** @var string */
    protected $_code = 'altapay';

    /** @var Awardit_Altapay_Helper_Data */
    protected $helper;

    /**
     * Constructor for class.
     */
    public function __construct()
    {
        $this->helper = Mage::helper('awardit_altapay');
    }

    /**
     * Get method code.
     * @return string
     */
    public function getCode(): string
    {
        return $this->_code;
    }

    /**
     * Get method type.
     * @return string
     */
    public function getMethodTypeName(): string
    {
        return ucfirst($this->_code);
    }

    /**
     * Get result data when setting Altapay as quote payment.
     * @param Registry $registry
     * @return array<array-key, MageQL\Type\FieldBuilder> Result fields
     */
    public function getMutateSetQuotePaymentFields(Registry $registry): array
    {
        $model = Mage::getSingleton('mageql_sales/quote');
        $payment = $model->getQuote()->getPayment();

        return [
            'result' => $this->field(
                "{$this->getMutationResultType()}!",
                'Result when setting payment method to AltaPay'
            )->setResolver('MageQL\\forwardResolver'),
            'sessionId' => $this->field('ID!', 'The session id for AltaPay')
                ->setResolver(function () use ($payment) {
                    $sessionId = $payment->getAdditionalInformation('sessionId');
                    return is_string($sessionId) ? $sessionId : '';
                }),
            'token' => $this->field('String!', 'Bearer token to use with Altapay')
                ->setResolver(function () use ($payment) {
                    $checkoutToken = $payment->getAdditionalInformation('checkoutToken');
                    return is_string($checkoutToken) ? $checkoutToken : '';
                }),
        ];
    }

    /**
     * Resolve result when setting Altapay as quote payment.
     * @param Mage_Sales_Model_Quote $quote
     * @param array $args
     * @param MageQL_Core_Model_Context $ctx
     * @param ResolveInfo $info
     * @return string Result code
     */
    public function mutateSetPaymentMethod(
        Mage_Sales_Model_Quote $quote,
        array $args,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): string {
        $model = Mage::getSingleton('mageql_sales/quote');
        try {
            $payment = $model->importPaymentData($quote, $this->getCode(), []);
            /** @var Awardit_Altapay_Model_Method_Altapay */
            $method = $payment->getMethodInstance();
            $method->selectSession($quote);
            $model->saveSessionQuote();
            return self::SUCCESS;
        } catch (Mage_Core_Exception $e) {
            $this->helper->log("Failed to set {$this->_code} on quote/{$quote->getId()}: {$e->getMessage()}");
            return $this->translatePaymentError($e);
        }
    }

    /**
     * Additional method fields for Altapay methods.
     * @param Registry $registry
     * @return array<array-key, MageQL\Type\FieldBuilder>
     */
    public function getPaymentMethodFields(Registry $registry): array
    {
        return [];
    }

    /**
     * Additional quote fields for Altapay methods.
     * @param Registry $registry
     * @return array<array-key, MageQL\Type\FieldBuilder>
     */
    public function getQuotePaymentFields(Registry $registry): array
    {
        return [];
    }
}
