<?php

use Awardit\MagentoPsr\Psr3\Logger;

/**
 * Helper class for Altapay.
 */
class Awardit_Altapay_Helper_Data extends Mage_Core_Helper_Abstract
{
    /** @const string */
    private const LOG_CHANNEL = 'altapay';

    /** @var Awardit_Altapay_Model_Api_Checkout */
    private Awardit_Altapay_Model_Api_Checkout $checkoutClient;

    /** @var Awardit_Altapay_Model_Api_Merchant */
    private Awardit_Altapay_Model_Api_Merchant $merchantClient;

    private Logger $psrLogger;

    public function __construct(
        ?Awardit_Altapay_Model_Api_Checkout $checkoutClient = null,
        ?Awardit_Altapay_Model_Api_Merchant $merchantClient = null
    ) {
        $this->psrLogger = new Logger(self::LOG_CHANNEL);
        if (!$checkoutClient) {
            $checkoutClient = new Awardit_Altapay_Model_Api_Checkout(null, null, $this->psrLogger);
        }
        if (!$merchantClient) {
            $merchantClient = new Awardit_Altapay_Model_Api_Merchant(null, $this->psrLogger);
        }
        $this->checkoutClient = $checkoutClient;
        $this->merchantClient = $merchantClient;
    }
    /**
     * Altapay Checkout Api Client
     * @return Awardit_Altapay_Model_Checkout_Client
     */
    public function checkout(): Awardit_Altapay_Model_Checkout_Client
    {
        return $this->checkoutClient->getClient();
    }

    /**
     * Altapay Merchant Api Client
     * @return Awardit_Altapay_Model_Api_Merchant
     */
    public function merchant(): Awardit_Altapay_Model_Api_Merchant
    {
        return $this->merchantClient;
    }

    /**
     * Altapay token for customer to use
     * @param string $sessionId
     * @return Awardit_Altapay_Model_Checkout_Type_TokenResponse
     */
    public function customerCheckoutToken(string $sessionId): Awardit_Altapay_Model_Checkout_Type_TokenResponse
    {
        return $this->checkoutClient->authenticate('CUSTOMER', $sessionId);
    }

    /**
     * Get active quote by reserved order increment id.
     * @param string $orderId
     * @return Mage_Sales_Model_Quote|null
     */
    public function getQuoteByOrderId(string $orderId): ?Mage_Sales_Model_Quote
    {
        $quote = Mage::getModel('sales/quote');
        $quote->load($orderId, 'reserved_order_id');
        return $quote->getIsActive() ? $quote : null;
    }

    /**
     * Write to log.
     * @param string $message Message to log.
     * @param 0|1|2|3|4|5|6|7|null $level Any Zend_Log level constant.
     */
    public function log(string $message, ?int $level = Zend_Log::DEBUG): void
    {
        Mage::log($message, $level, self::LOG_CHANNEL);
    }

    public function getPsrLogger(): Logger
    {
        return $this->psrLogger;
    }

    public function throw(string $message, int $code = 500): void
    {
        $this->psrLogger->error($message);
        throw new Awardit_Altapay_Exception($message, $code);
    }
}
