<?php

use GraphQL\Error\ClientAware;
use Psr\Log\LoggerInterface;

class Awardit_Altapay_Model_Method_Altapay extends Mage_Payment_Model_Method_Abstract
{
    /** @var mixed */
    protected $_code = 'altapay';
    /** @var bool */
    protected $_isGateway = true;
    /** @var mixed */
    protected $_canAuthorize = true;
    /** @var mixed */
    protected $_canCapture = true;
    /** @var mixed */
    protected $_canRefund = true;
    /** @var mixed */
    protected $_canRefundInvoicePartial = true;
    /** @var mixed */
    protected $_canVoid = true;
    /** @var mixed */
    protected $_canUseInternal = false;
    /** @var mixed */
    protected $_canUseCheckout = true;
    /** @var mixed */
    protected $_canUseForMultishipping = false;
    /** @var mixed */
    protected $_canFetchTransactionInfo = false;
    /** @var mixed */
    protected $_canManageRecurringProfiles = false;

    /** @var array */
    protected $_dirty = [];

    /** @var Awardit_Altapay_Helper_Data */
    protected $helper;
    protected LoggerInterface $logger;

    /**
     * Constructor for class.
     */
    public function __construct()
    {
        $this->helper = Mage::helper('awardit_altapay');
        $this->logger = $this->helper->getPsrLogger();
    }

    /**
     * Check whether payment method can be used.
     * @param Mage_Sales_Model_Quote|null $quote
     * @return bool
     */
    public function isAvailable($quote = null)
    {
        if (!parent::isAvailable($quote)) {
            return false;
        }
        if (empty($quote) || empty($quote->getId()) || empty($quote->getItemsCount())) {
            return false;
        }
        if (!$this->canUseForCountry($quote->getBillingAddress()->getCountry())) {
            return false;
        }
        if (!$this->canUseForCurrency($quote->getQuoteCurrencyCode())) {
            return false;
        }
        if (!$this->canUseCheckout()) {
            return false;
        }
        $total = $quote->getBaseSubtotal() + $quote->getShippingAddress()->getBaseShippingAmount();
        if ($total < 0.0001) {
            return false;
        }
        return true;
    }

    /**
     * Configure session for checkout (stores sessionId for future use) when method is selected.
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @return $this
     * @psalm-suppress UnusedVariable, TypeDoesNotContainType, NoValue, RedundantCondition, InvalidCast
     */
    public function selectSession(Mage_Sales_Model_Quote $quote): self
    {
        $payment = $quote->getPayment();
        $client = $this->helper->checkout();
        $quote->reserveOrderId();

        /** @var string */
        $sessionId = $payment->getAdditionalInformation('sessionId');
        /** @var string */
        $checkoutToken = $payment->getAdditionalInformation('checkoutToken');

        // @todo: Temporary fix to enforce new session when quote is changed
        // - Long term fix; use observer and/or checksum to evaluate if new session is needed
        // - Unique orderId per session - incrementing suffix?
        // $sessionId = $checkoutToken = '';

        // Check if existing session still available
        if (!empty($sessionId)) {
            try {
                $session = $client->getSession($sessionId);
                $this->logger->debug(sprintf(
                    'Read session/%s from Altapay for quote/%s: %s',
                    $sessionId,
                    $quote->getId(),
                    json_encode($session),
                ));
                // Generate new token
                $checkoutToken = $this->helper->customerCheckoutToken($session->sessionId)->token;
                $payment->setAdditionalInformation('checkoutToken', $checkoutToken);
                $payment->save();
            } catch (Awardit_Altapay_Model_Checkout_Exception_NotFoundException $e) {
                $this->logger->warning("Expired Altapay session {$sessionId} for Quote {$quote->getId()}");
                $sessionId = $checkoutToken = '';
            }
        }

        // Create new checkout session
        if (empty($sessionId) || empty($checkoutToken)) {
            $sessionConf = $this->getCheckoutConfiguration($quote->getStore());
            $session = $client->createSession(new Awardit_Altapay_Model_Checkout_Type_Session($sessionConf));
            $checkoutToken = $this->helper->customerCheckoutToken($session->sessionId)->token;
            $this->logger->debug(sprintf(
                "Create Altapay session/%s for quote/%s: %s",
                $session->sessionId,
                $quote->getId(),
                json_encode($sessionConf)
            ));

            // Store sessionId and some other data for later use
            $payment->setAdditionalInformation('sessionId', $session->sessionId);
            $payment->setAdditionalInformation('checkoutToken', $checkoutToken);
            $payment->setAdditionalInformation('checksum', null);
            $payment->setAdditionalInformation('method', $this->getCode());
            $payment->save();
        }

        $this->updateSession($quote);

        return $this;
    }

    /**
     * Updates session based on quote. Triggered by observer when quote is changed.
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @return $this
     */
    public function updateSession(Mage_Sales_Model_Quote $quote): self
    {
        if (!$quote->getIsActive()) {
            return $this; // No update of closed quotes
        }
        $payment = $quote->getPayment();
        if ($payment->getMethod() != 'altapay') {
            return $this;
        }
        /** @var string */
        $sessionId = $payment->getAdditionalInformation('sessionId');
        if (empty($sessionId)) {
            return $this;
        }
        $original = $quote->getReservedOrderId();
        $quote->reserveOrderId(); // Ensure unique order increment
        if ($quote->getReservedOrderId() != $original) {
            $this->logger->debug("Updated reserved id {$original} to {$quote->getReservedOrderId()}");
        }
        // Check if Altapay needs to be updated
        $sessionData = $this->getCheckoutSessionFromQuote($quote);
        $sessionJson = json_encode($sessionData);
        $sessionChecksum = md5($sessionJson);
        if ($sessionChecksum == $payment->getAdditionalInformation('checksum')) {
            $this->logger->debug(
                "Intended to update Altapay session/{$sessionId} " .
                "for quote/{$quote->getId()}: {$sessionJson}, but no update required. Checksum is the same"
            );
            return $this;
        }

        // Update Altapay with Quote data
        $client = $this->helper->checkout();
        $client->updateSession($sessionId, new Awardit_Altapay_Model_Checkout_Type_Session($sessionData));
        $payment->setAdditionalInformation('checksum', $sessionChecksum);
        $payment->save();
        $this->logger->debug("Updated Altapay session/{$sessionId} for quote/{$quote->getId()}: {$sessionJson}");
        return $this;
    }

    /**
     * Create a Checkout Session configuration.
     * @return array Session configuration
     */
    private function getCheckoutConfiguration(Mage_Core_Model_Store $store): array
    {
        // @todo: Enable more fields as needed
        return $this->filter([
            'callbacks' => $this->filter([
                'success' => ['type' => 'FUNCTION'],
                'failure' => ['type' => 'FUNCTION'],
                // 'redirect' => '',
                'notification' => (string)$store->getConfig('payment/altapay/cb_notification'),
                'verifyOrder' => (string)$store->getConfig('payment/altapay/cb_verify'),
                // 'formStyling' => '',
                'bodyFormat' => 'JSON',
            ]),
            'configuration' => $this->filter([
                'paymentType' => 'PAYMENT',
                'paymentDisplayType' => 'SCRIPT',
                // 'agreement' => [],
                'autoCapture' => $store->getConfig('payment/altapay/payment_action') == 'authorize_capture',
                // Must match Altapay config, contact them
                'shopName' => (string)$store->getConfig('payment/altapay/shop_name'),
                // Default country, address country override when set
                'country' => Mage::helper('core')->getDefaultCountry($store),
                'language' => substr($store->getConfig('general/locale/code') ?? 'en', 0, 2),
                // 'ccToken' => [],
            ]),
        ]);
    }

    /**
     * Create a Checkout Session from Quote data.
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @return array Session quote data
     */
    private function getCheckoutSessionFromQuote(Mage_Sales_Model_Quote $quote): array
    {
        return ['order' => $this->filter([
            'orderId' => (string)$quote->getReservedOrderId(),
            'amount' => [
                'value' =>  (float)$quote->getGrandTotal(),
                'currency' => $quote->getQuoteCurrencyCode(),
            ],
            'orderLines' => $this->getOrderLines($quote),
            'customer' => $this->getCustomer($quote),
            'transactionInfo' => [
                'quoteId' => $quote->getId(),
                'storeCode' => $quote->getStore()->getCode(),
            ],
        ])];
    }

    /**
     * Create a Checkout Session from Order data.
     * @param Mage_Sales_Model_Order $order Order to checkout
     * @return array Session order data
     */
    private function getCheckoutSessionFromOrder(Mage_Sales_Model_Order $order): array
    {
        return ['order' => $this->filter([
            'orderId' => (string)$order->getIncrementId(),
            'amount' => [
                'value' => (float)$order->getGrandTotal(),
                'currency' => $order->getOrderCurrencyCode(),
            ],
            'orderLines' => $this->getOrderLines($order),
            'customer' => $this->getCustomer($order),
            'transactionInfo' => [
                'quoteId' => $order->getQuoteId(),
                'storeCode' => $order->getStore()->getCode(),
            ],
        ])];
    }

    /**
     * Get shipping fee item from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress MismatchingDocblockParamType
     */
    private function getShippingItem(Mage_Core_Model_Abstract $source): array
    {
        $source = $source instanceof Mage_Sales_Model_Quote ? $source->getShippingAddress() : $source;
        $unitPrice = round($source->getShippingAmount(), 2);
        return $this->filter($unitPrice == 0 ? [] : [
            'goodsType' => 'shipment',
            'itemId' => 'shipment',
            'description' => $source->getShippingDescription(),
            'quantity' => 1,
            'unitPrice' => $unitPrice,
            'taxAmount' => round($source->getShippingTaxAmount() + $source->getShippingHiddenTaxAmount(), 2),
            'discountPercent' => 0, // @todo: Not sure about the discount stuff
        ]);
    }

    /**
     * Get payment fee item from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress MismatchingDocblockParamType
     */
    private function getPaymentFeeItem(Mage_Core_Model_Abstract $source): array
    {
        if ($source instanceof Mage_Sales_Model_Quote) {
            $totals = $source->getTotals();
            if (!isset($totals['payment_fee'])) {
                return [];
            }
            $fee = $totals['payment_fee'];
            $excl = round($fee->getValueExclTax(), 2);
            $tax = round($fee->getValueInclTax() - $fee->getValueExclTax(), 2);
            $name = $fee->getTitle();
        } else {
            $excl = round($source->getPaymentFeeAmount(), 2);
            $tax = round($source->getPaymentFeeTaxAmount(), 2);
            $name = $source->getPaymentFeeTitle();
        }
        return $excl > 0 ? $this->filter([
            'goodsType' => 'handling',
            'itemId' => 'payment',
            'description' => $name ?: 'Payment fee',
            'quantity' => 1,
            'unitPrice' => $excl,
            'taxAmount' => $tax,
            'discountPercent' => 0, // @todo: Not sure about the discount stuff
        ]) : [];
    }

    /**
     * Get customer info from Order or Quote.
     * Source priority: 1) Order|Quote 2) Billing address 3) Shipping address
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress RedundantConditionGivenDocblockType
     */
    private function getCustomer(Mage_Core_Model_Abstract $source): array
    {
        $customer = [];
        if ($shippingAddress = $source->getShippingAddress()) {
            $customer = array_merge($customer, $this->filter([
                'firstName' => $shippingAddress->getFirstname(),
                'lastName' => $shippingAddress->getLastname(),
                'email' => $shippingAddress->getEmail(),
                'phoneNumber' => $shippingAddress->getTelephone(),
                'shippingAddress' => $shippingAddress->validate() === true ? $this->filter([
                    'firstName' => $shippingAddress->getFirstname(),
                    'lastName' => $shippingAddress->getLastname(),
                    'street' => $shippingAddress->getStreetFull(),
                    'city' => $shippingAddress->getCity(),
                    'region' => $shippingAddress->getRegion(),
                    'country' => $shippingAddress->getCountry(),
                    'zipCode' => $shippingAddress->getPostcode(),
                ]) : null,
            ]));
        }
        if ($billingAddress = $source->getBillingAddress()) {
            $customer = array_merge($customer, $this->filter([
                'firstName' => $billingAddress->getFirstname(),
                'lastName' => $billingAddress->getLastname(),
                'email' => $billingAddress->getEmail(),
                'phoneNumber' => $billingAddress->getTelephone(),
                'billingAddress' => $billingAddress->validate() === true ? $this->filter([
                    'firstName' => $billingAddress->getFirstname(),
                    'lastName' => $billingAddress->getLastname(),
                    'street' => $billingAddress->getStreetFull(),
                    'city' => $billingAddress->getCity(),
                    'region' => $billingAddress->getRegion(),
                    'country' => $billingAddress->getCountry(),
                    'zipCode' => $billingAddress->getPostcode(),
                ]) : null,
            ]));
        }
        $customer = array_merge($customer, $this->filter([
            'firstName' => $source->getCustomerFirstname(),
            'lastName' => $source->getCustomerLastname(),
            'email' => $source->getCustomerEmail(),
        ]));
        return $this->filter($customer);
    }

    /**
     * Get order items from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress MismatchingDocblockParamType, UndefinedMagicMethod
     */
    private function getOrderItems(Mage_Core_Model_Abstract $source): array
    {
        $isOrder = $source instanceof Mage_Sales_Model_Order;
        return array_map(function ($item) use ($source, $isOrder) {
            $product = $item->getProduct();
            $product->setStoreId($source->getStoreId())->load($product->getId());
            $type = $item->getIsVirtual() ? 'digital' : 'item';
            $images = $product->getMediaGalleryImages();
            if (!$isOrder) {
                $item->calcTaxAmount();
            }
            return $this->filter([
                'itemId' => $product->getSku(),
                'description' => $product->getName(),
                'quantity' => $isOrder ? $item->getQtyOrdered() : $item->getQty(),
                'unitPrice' => $isOrder ? $item->getPrice() : $item->getConvertedPrice(),
                'taxAmount' =>  $isOrder ? $item->getTaxAmount() : $item->getTaxBeforeDiscount(),
                'taxPercent' => (float)$item->getTaxPercent(),
                'discountPercent' => (float)$item->getDiscountPercent(),
                'goodsType' => $type,
                'imageUrl' => $images->getSize() > 0 ? $images->getFirstItem()->getUrl() : null,
                'productUrl' => $product->getUrlInStore(),
            ]);
        }, $source->getAllVisibleItems());
    }

    /**
     * Get discount items from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     */
    private function getDiscountItem(Mage_Core_Model_Abstract $source): array
    {
        if ($source instanceof Mage_Sales_Model_Quote) {
            $totals = $source->getTotals();
            if (!isset($totals['discount']) || $totals["discount"]->getValue() === 0) {
                return [];
            }
            $amount = $totals["discount"]->getValue();
        } else {
            $amount = $source->getDiscountAmount();
        }
        $amount = round($amount, 2);

        return $amount == 0 ? [] : $this->filter([
            'goodsType' => 'discount',
            'itemId' => 'discount',
            'description' => 'Discount',
            'quantity' => 1,
            'unitPrice' => $amount,
            'taxAmount' => 0, // @todo: Not sure what to do here
            'discountPercent' => 0, // @todo: Not sure about the discount stuff
        ]);
    }

    /**
     * Get Retain24 discount items from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     */
    private function getRetain24Item(Mage_Core_Model_Abstract $source): array
    {
        if ($source instanceof Mage_Sales_Model_Quote) {
            $totals = $source->getTotals();
            if (!isset($totals['retain24']) || $totals["retain24"]->getValue() === 0) {
                return [];
            }
            $amount = $totals["retain24"]->getValue();
        } else {
            $amount = $source->getRetain24Amount();
        }
        $amount = round($amount, 2);

        return $amount == 0 ? [] : $this->filter([
            'goodsType' => 'discount',
            'itemId' => 'discount',
            'description' => 'Discount',
            'quantity' => 1,
            'unitPrice' => $amount,
            'taxAmount' => 0, // @todo: Not sure what to do here
            'discountPercent' => 0, // @todo: Not sure about the discount stuff
        ]);
    }

    /**
     * Get all order lines (including fees) from Order or Quote.
     * @param Mage_Sales_Model_Quote|Mage_Sales_Model_Order $source
     * @return array
     * @psalm-suppress MismatchingDocblockParamType
     */
    private function getOrderLines(Mage_Core_Model_Abstract $source): array
    {
        return array_values($this->filter(array_merge($this->getOrderItems($source), [
            $this->getShippingItem($source),
            $this->getPaymentFeeItem($source),
            $this->getDiscountItem($source),
            $this->getRetain24Item($source),
        ])));
    }

    /**
     * Create Magento Order, update Altapay session accordingly.
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @return Mage_Sales_Model_Order
     * @throws Awardit_Altapay_Exception if unable to create order and/or update session
     * @psalm-suppress PossiblyUnusedReturnValue, InvalidReturnType
     */
    public function createOrder(
        Mage_Sales_Model_Quote $quote,
        Awardit_Altapay_Model_Type_NotificationCallback $notification
    ): Mage_Sales_Model_Order {
        $logContext = [
            'quote' => $quote->getId(),
            'orderId' => (string)$quote->getReservedOrderId(),
        ];

        $this->logger->debug('Creating order for quote/{quote}', $logContext);
        if (!$quote->getIsActive()) {
            $this->helper->throw("Quote {$quote->getId()} not active.", 400);
        }
        $payment = $quote->getPayment();
        if ($payment->getMethod() != 'altapay') {
            $this->helper->throw("Quote {$quote->getId()} with wrong payment method: {$payment->getMethod()}.", 400);
        }
        /** @var string */
        $sessionId = $payment->getAdditionalInformation('sessionId');
        if (empty($sessionId)) {
            $this->helper->throw("Quote {$quote->getId()} have no session.", 400);
        }
        $logContext['sessionId'] = $sessionId;
        $client = $this->helper->checkout();
        $client->getSession($sessionId); // Throws error if missing

        // Check if Quote has invalid changes
        $sessionData = $this->getCheckoutSessionFromQuote($quote);
        $sessionJson = json_encode($sessionData);
        $sessionChecksum = md5($sessionJson);
        $logContext['session'] = $sessionJson;
        if ($sessionChecksum != $payment->getAdditionalInformation('checksum')) {
            $logContext['checksum'] = [
                'expected' => $payment->getAdditionalInformation('checksum'),
                'actual' => $sessionChecksum,
            ];
            $this->logger->error('Invalid changes on quote/{quote}', $logContext);
            $this->helper->throw("Invalid changes on quote/{$quote->getId()}");
        }

        // Set dummy address
        $store = $quote->getStore();

        /** @var Mage_Sales_Model_Quote_Address */
        $billingAddress = $quote->getBillingAddress();
        if (
            $billingAddress->validate() !== true &&
            $quote->isVirtual() &&
            Mage::helper("mageql_sales")->getAllowPlaceholderBillingAddress($store)
        ) {
            $defaultCountryId = $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY) ?? '';
            $billingAddress
                ->setAddressType(Mage_Sales_Model_Quote_Address::TYPE_BILLING)
                ->setFirstname($billingAddress->getFirstname() ?: "-")
                ->setLastname($billingAddress->getLastname() ?: "-")
                ->setStreet($billingAddress->getStreet(-1) ?: ["-"])
                ->setPostcode($billingAddress->getPostcode() ?: "-")
                ->setTelephone($billingAddress->getTelephone() ?: "-")
                ->setCity($billingAddress->getCity() ?: "-")
                ->setCountryId($billingAddress->getCountryId() ?: $defaultCountryId);
        }

        // Finalize Quote
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        try {
            // Create Magento Order
            $service = Mage::getModel('sales/service_quote', $quote);
            $service->submitAll();

            $order = $service->getOrder();
            $payment = $order->getPayment();

            $this->logger->info('Created order/{orderId} for quote/{quote}', $logContext);

            // Close Quote
            $quote->setIsActive(0);
            $quote->save();

            if ($payment) {
                $this->addNotificationToPayment($payment, $notification);
                $payment->save();
            }

            return $order;
        } catch (Mage_Core_Exception $e) {
            $logContext['exception'] = $e;
            $logContext['error'] = $e->getMessage();
            $this->logger->error('Failed creating order for quote/{quote}; {error}', $logContext);
            if ($e instanceof ClientAware) {
                throw $e; // Already user friendly
            }
            $this->helper->throw("Failed creating order for quote/{$quote->getId()}: {$e->getMessage()}");
        }
    }

    private function addNotificationToPayment(
        Mage_Sales_Model_Order_Payment $payment,
        Awardit_Altapay_Model_Type_NotificationCallback $notification
    ): void {
        $this->addNotificationToPaymentRecursive($payment, [
            'requiresCapture' => $notification->requiresCapture,
            'status' => $notification->status,
            'sessionId' => $notification->sessionId,
            'cardInformation' => $notification->cardInformation,
            'method' => $notification->method,
            'payment' => $notification->payment,
            'riskAnalysis' => $notification->riskAnalysis,
        ]);
        /** @psalm-suppress InvalidCast, InvalidArgument */
        $payment->setTransactionAdditionalInfo(
            Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS,
            array_filter(array_merge(
                $payment->getAdditionalInformation() ?? [],
                $payment->getTransactionAdditionalInfo() ?? []
            ), function ($item) {
                return !empty($item) && is_scalar($item);
            })
        );
    }

    private function addNotificationToPaymentRecursive(
        Mage_Sales_Model_Order_Payment $payment,
        array $data,
        string $prefix = ''
    ): void {
        foreach ($data as $key => $value) {
            if (empty($value)) {
                continue;
            } elseif (is_array($value)) {
                $this->addNotificationToPaymentRecursive($payment, $value, "{$prefix}{$key}.");
            } elseif (is_scalar($value)) {
                $payment->setAdditionalInformation("{$prefix}{$key}", $value);
                $payment->setTransactionAdditionalInfo("{$prefix}{$key}", (string)$value);
            }
        }
    }

    /**
     * Handle a webhhook "notification" request from Altapay.
     * @psalm-suppress PossiblyUnusedReturnValue
     */
    public function updateOrder(
        Mage_Sales_Model_Order $order,
        Awardit_Altapay_Model_Type_NotificationCallback $notification
    ): Mage_Sales_Model_Order_Payment {
        /** @var Mage_Sales_Model_Order_Payment */
        $payment = $order->getPayment();
        $this->logger->debug("Updating order {$order->getIncrementId()}, setting status: {$notification->status}");

        switch (strtolower($notification->status)) {
            case 'succeeded':
                $transaction = $payment->getAuthorizationTransaction();
                if (empty($transaction)) {
                    return $payment; // Not able to handle?
                }
                if ($transaction->getIsClosed()) {
                    return $payment; // Already handled
                }
                $payment->setTransactionId($transaction->getTxnId());
                $this->addNotificationToPayment($payment, $notification);
                $payment->setNotificationResult(true);
                $payment->authorize(false, $notification->order['amount']);
                $order->setCanSendNewEmailFlag(true);
                $this->logger->debug("Updated order {$order->getIncrementId()}, confirming {$transaction->getTxnId()}");

                try {
                    $order->queueNewOrderEmail();
                } catch (Throwable $e) {
                    $this->logger->error(sprintf(
                        "%s(%s): Exception sending email %s",
                        __METHOD__,
                        $order->getId(),
                        $e->getMessage()
                    ));
                }

                // For some reason we need a "order" transaction to capture from admin
                $orderTransaction = $payment->lookupTransaction(
                    "{$transaction->getTxnId()}-order",
                    Mage_Sales_Model_Order_Payment_Transaction::TYPE_ORDER
                );
                if (empty($orderTransaction)) {
                    Mage::getModel('sales/order_payment_transaction')
                        ->setTxnId("{$transaction->getTxnId()}-order")
                        ->setOrderPaymentObject($payment)
                        ->setTxnType(Mage_Sales_Model_Order_Payment_Transaction::TYPE_ORDER)
                        ->setParentTxnId($transaction->getTxnId())
                        ->save();
                }

                break;
            case 'cancelled':
            case 'declined':
            case 'error':
            case 'failed':
                // Checkout failed or cancelled by customer after initiated payment
                if (!$order->canCancel()) {
                    $this->logger->warning("Can not cancel order {order} with status {status}", [
                        'order' => $order->getIncrementId(),
                        'status' => $order->getStatus(),
                    ]);
                    return $payment;
                }
                // Cancel order
                $this->addNotificationToPayment($payment, $notification);
                $payment->setIsTransactionClosed(true);
                $payment->save();
                $order->cancel("{$notification->merchantErrorMessage} ({$notification->status})");
                $order->save();
                $this->logger->debug("Cancelled order {$order->getIncrementId()}.");

                // Re-open quote, if still exist
                $quoteId = $order->getQuoteId();
                $quote = Mage::getModel('sales/quote')->load($quoteId);
                if (!$quote->getId()) {
                    break;
                }

                // Reset quote payment session
                $quotePayment = $quote->getPayment();
                $quotePayment->setAdditionalInformation('sessionId', null);
                $quotePayment->setAdditionalInformation('checkoutToken', null);
                $quotePayment->save();

                // Re-open quote with new order number
                $quote->setIsActive(1);
                $quote->setReservedOrderId(''); // Reset reserved
                $quote->save();
                $this->logger->debug("Re-opened quote {$quote->getId()}.");
                break;
            case 'unknown':
                // @todo: What to do on these?
                $payment->setIsTransactionDenied(true);
                $payment->setIsTransactionClosed(true);
                break;
            case 'new':
            case 'pending':
                // @todo: not sure what to do here
                $payment->setIsTransactionPending(true);
                break;
            default:
                $this->helper->throw("Unhandled Altapay status: {$notification->status}");
        }

        $payment->save();
        $order->save();

        return $payment;
    }

    /**
     * Handle a webhhook "notification" request from Altapay.
     * @psalm-suppress PossiblyUnusedReturnValue
     */
    public function updateQuote(
        Mage_Sales_Model_Quote $quote,
        Awardit_Altapay_Model_Type_NotificationCallback $notification
    ): Mage_Sales_Model_Quote_Payment {
        $payment = $quote->getPayment();
        $this->logger->debug("Updating quote {$quote->getReservedOrderId()}, status: {$notification->status}");

        switch (strtolower($notification->status)) {
            case 'succeeded':
                $this->logger->warning("Quote {$quote->getReservedOrderId()} should not be succeeded");
                break;
            case 'cancelled':
            case 'declined':
            case 'error':
            case 'failed':
                $quote->setReservedOrderId(''); // Reset reserved
                $this->logger->debug("Reset quote reserved id {$quote->getId()}.");
                // Reset quote payment session
                $payment = $quote->getPayment();
                $payment->setAdditionalInformation('sessionId', null);
                $payment->setAdditionalInformation('checkoutToken', null);
                break;
            case 'unknown':
                // @todo: What to do on these?
                break;
            case 'new':
            case 'pending':
                // @todo: not sure what to do here
                break;
            default:
                $this->helper->throw("Unhandled Altapay status: {$notification->status}");
        }

        $payment->save();
        $quote->save();
        return $payment;
    }

    /**
     * Authorize payment method.
     * @param Mage_Sales_Model_Order_Payment $payment
     * @param float $amount
     * @return $this
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function authorize(Varien_Object $payment, $amount): self
    {
        parent::authorize($payment, $amount); // Checks availability

        $order = $this->getOrder($payment);

        $this->logger->debug("Authorization for order/{order}.", [
            'order' => $order->getIncrementId(),
            'amount' => $amount,
        ]);

        // @todo: Verify Altapay?

        /** @var string */
        $sessionId = $payment->getAdditionalInformation('sessionId');
        $payment->setTransactionId($sessionId);
        $payment->setIsTransactionDenied(false);
        $payment->setIsTransactionClosed(false);
        $payment->setIsTransactionPending(true);
        $payment->setIsTransactionApproved(true);
        $payment->save();

        return $this;
    }

    /**
     * Capture payment, either directly or on previously authorized payment.
     * @param Mage_Sales_Model_Order_Payment $payment
     * @param float $amount
     * @return $this
     * @throws Awardit_Altapay_Exception on failure
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function capture(Varien_Object $payment, $amount): self
    {
        parent::capture($payment, $amount); // Checks availability

        $merchant = $this->helper->merchant();
        $order = $this->getOrder($payment);
        /** @var string */
        $sessionId = $payment->getAdditionalInformation('sessionId');
        $transactionId = $this->getPaymentId($payment);
        $payment->setTransactionId("{$sessionId}-capture");
        try {
            $merchant->captureReservation($transactionId, $amount); // @todo save additional info?
            $order->setCanSendNewEmailFlag(true);
            $this->logger->info("Capture complete for order/{order}.", [
                'transactionId' => $transactionId,
                'order' => $order->getIncrementId(),
                'amount' => $amount,
            ]);
        } catch (Throwable $e) {
            $this->logger->error("Capture failed for order/{order}.", [
                'transactionId' => $transactionId,
                'order' => $order->getIncrementId(),
            ]);
            $this->helper->throw("Altapay capture failed: {$e->getMessage()}.");
        }

        return $this;
    }

    /**
     * Cancel authorization.
     * @param Mage_Sales_Model_Order_Payment $payment
     * @return $this
     * @throws Awardit_Altapay_Exception on failure
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function cancel(Varien_Object $payment): self
    {
        parent::cancel($payment); // Checks availability

        $payment->setIsTransactionDenied(true);
        $payment->setIsTransactionClosed(true);

        $merchant = $this->helper->merchant();
        $order = $this->getOrder($payment);
        $transactionId = $this->getPaymentId($payment);

        try {
            $merchant->releaseReservation($transactionId); // @todo save additional info?
            $this->logger->info("Cancel complete for order/{order}.", [
                'transactionId' => $transactionId,
                'order' => $order->getIncrementId(),
            ]);
        } catch (Throwable $e) {
            $this->logger->error("Cancel failed for order/{order}.", [
                'transactionId' => $transactionId,
                'order' => $order->getIncrementId(),
            ]);
            $this->helper->throw("Altapay cancel failed: {$e->getMessage()}.");
        }

        $payment->save();
        return $this;
    }

    /**
     * Void, same as cancel.
     * @param  Mage_Sales_Model_Order_Payment $payment
     * @return $this
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function void(Varien_Object $payment): self
    {
        parent::void($payment); // Checks availability
        return $this->cancel($payment);
    }

    /**
     * Refund specified amount for payment
     * @param Mage_Sales_Model_Order_Payment $payment
     * @param float $amount
     * @return $this
     * @throws Awardit_Altapay_Exception on failure
     * @psalm-suppress MoreSpecificImplementedParamType, MethodSignatureMismatch
     */
    public function refund(Varien_Object $payment, $amount): self
    {
        parent::refund($payment, $amount); // Checks availability

        $payment->setIsTransactionDenied(true);
        $payment->setIsTransactionClosed(true);

        $merchant = $this->helper->merchant();
        $order = $this->getOrder($payment);
        $transactionId = $this->getPaymentId($payment);

        try {
            $merchant->refundCapturedReservation($transactionId, $amount); // @todo save additional info?
            $this->logger->info("Refund complete for order/{order}.", [
                'transactionId' => $transactionId,
                'order' => $order->getIncrementId(),
            ]);
        } catch (Throwable $e) {
            $this->logger->error("Refund failed for order/{order}.", [
                'transactionId' => $transactionId,
                'order' => $order->getIncrementId(),
            ]);
            $this->helper->throw("Altapay refund failed: {$e->getMessage()}.");
        }

        $payment->save();
        return $this;
    }

    /**
     * Get Checkout paymentId (used as Metchant transactionId) from payment
     * @throws Awardit_Altapay_Exception if paymentId can not be found
     */
    private function getPaymentId(Mage_Sales_Model_Order_Payment $payment): string
    {
        /** @var string */
        $transactionId = $payment->getAdditionalInformation('payment.gateway.paymentId');
        if (empty($transactionId)) {
            $order = $this->getOrder($payment);
            $this->logger->error("Could not resolve paymentId.", [
                'order' => $order->getIncrementId(),
            ]);
            $this->helper->throw("Could not resolve Altapay paymentId.");
        }
        return $transactionId;
    }

    /**
     * Sometimes Order is not pre-handled on Payment. This method ensures it is.
     * @param Mage_Sales_Model_Order_Payment $payment Magento payment
     * @return Mage_Sales_Model_Order
     */
    private function getOrder(Mage_Sales_Model_Order_Payment $payment): Mage_Sales_Model_Order
    {
        $order = $payment->getOrder();
        if (empty($order)) {
            $order = Mage::getModel('sales/order');
            $order->load($payment->getParentId());
            $payment->setOrder($order);
        }
        return $order;
    }

    // Don't filter booleans
    private function filter(array $source): array
    {
        return array_filter($source, function ($item) {
            return is_bool($item) || !empty($item);
        });
    }
}
