<?php

use Psr\Log\LoggerInterface;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Normalizer\ArrayDenormalizer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;

class Awardit_Altapay_CallbackController extends Mage_Core_Controller_Front_Action
{
    protected ?Serializer $serializer = null;
    protected LoggerInterface $logger;
    protected Awardit_Altapay_Helper_Data $helper;

    protected function _construct(): void
    {
        parent::_construct();
        $this->helper = Mage::helper('awardit_altapay');
        $this->logger = $this->helper->getPsrLogger();
    }

    /**
     * Get serializer
     * @return Serializer
     */
    protected function getSerializer(): Serializer
    {
        if (empty($this->serializer)) {
            $normalizers = [
                new ArrayDenormalizer(),
                new ObjectNormalizer(),
            ];
            $this->serializer = new Serializer($normalizers, [new JsonEncoder()]);
        }
        return $this->serializer;
    }

    /**
     * @param int $code
     * @param mixed $body
     * @return Zend_Controller_Response_Abstract
     * @throws Zend_Controller_Response_Exception
     */
    protected function jsonResponse($code, $body)
    {
        return $this->getResponse()
            ->setHttpResponseCode($code)
            ->setHeader('Content-Type', 'application/json; charset=utf-8', true)
            ->setBody(json_encode($body, Mage::getIsDeveloperMode() ? JSON_PRETTY_PRINT : 0));
    }

    /**
     * Failure callback from Altapay
     * @return Zend_Controller_Response_Abstract
     */
    public function failureAction()
    {
        return $this->jsonResponse(200, [
            'result' => 'ok',
        ]);
    }

    /**
     * Notification callback from Altapay
     * @return Zend_Controller_Response_Abstract
     */
    public function notificationAction()
    {
        $req = $this->getRequest();
        $logContext = [
            'action' => __METHOD__,
            'method' => $req->getMethod(),
            'uri' => $req->getRequestUri(),
            'body' => (string)$req->getRawBody(),
        ];

        if ('POST' !== $req->getMethod()) {
            $this->logger->warning("{action} method not allowed", $logContext);
            return $this->jsonResponse(405, [
                'errorMessage' => 'Method not allowed',
            ]);
        }

        try {
            /** @var Awardit_Altapay_Model_Type_NotificationCallback|null $body */
            $body = $this->getSerializer()->deserialize(
                $req->getRawBody(),
                'Awardit_Altapay_Model_Type_NotificationCallback',
                'json'
            );
        } catch (Throwable $e) {
            $this->logger->warning("{action} malformed request: {$e->getMessage()}", $logContext);
            return $this->jsonResponse(400, [
                'errorMessage' => 'Malformed request',
            ]);
        }

        if (empty($body) || empty($body->order) || empty($body->order['orderId'])) {
            $this->logger->warning("{action} missing order.orderId", $logContext);
            return $this->jsonResponse(400, [
                'errorMessage' => 'Malformed request',
            ]);
        }

        $logContext['orderId'] = $body->order['orderId'];

        if ($body->status == 'new') {
            $this->logger->debug("{action} not created yet: {orderId}", $logContext);
            return $this->jsonResponse(200, ['result' => 'ok']);
        }

        $verified = $this->helper->verifySessionToken($body->sessionId, $body->order);
        if (!$verified) {
            $this->logger->error("Integrity check failed: {orderId}", $logContext);
            return $this->jsonResponse(400, [
                'errorMessage' => 'Unverified',
            ]);
        }

        $order = $this->helper->getOrderByOrderId($body->order['orderId']);
        $quote = isset($body->order['paymentMetadata']['quoteId'])
            ? $this->helper->getQuoteByQuoteId($body->order['paymentMetadata']['quoteId'])
            : null;

        if (!$order && !$quote) {
            $this->logger->warning("{action} orderId not found: {orderId}", $logContext);
            return $this->jsonResponse(404, [
                'errorMessage' => 'Order id not found',
            ]);
        }

        // Update Magento order or quote
        try {
            if ($order) {
                /** @var Mage_Sales_Model_Order_Payment */
                $payment = $order->getPayment();
                /** @var Awardit_Altapay_Model_Method_Altapay */
                $method = $payment->getMethodInstance();
                $method->updateOrder($order, $body);
                $this->logger->info("{action} updated order {orderId}", $logContext);
            }
            if ($quote) {
                $payment = $quote->getPayment();
                /** @var Awardit_Altapay_Model_Method_Altapay */
                $method = $payment->getMethodInstance();
                $method->updateQuote($quote, $body);
                $this->logger->info("{action} updated quote {orderId}", $logContext);
            }
            return $this->jsonResponse(200, [
                'result' => 'ok',
            ]);
        } catch (Awardit_Altapay_Exception $e) {
            $this->logger->error("{action} error: {$e->getMessage()}", $logContext);
            return $this->jsonResponse($e->getCode(), [
                'errorMessage' => $e->getHttpMessage(),
            ]);
        } catch (Throwable $e) {
            $this->logger->error("{action} error: {$e->getMessage()}", $logContext);
            return $this->jsonResponse(500, [
                'errorMessage' => 'Server error',
            ]);
        }
    }

    /**
     * Verify callback from Altapay
     * @return Zend_Controller_Response_Abstract
     */
    public function verifyAction()
    {
        $req = $this->getRequest();
        $logContext = [
            'action' => __METHOD__,
            'method' => $req->getMethod(),
            'uri' => $req->getRequestUri(),
            'body' => (string)$req->getRawBody(),
        ];

        if ('POST' !== $req->getMethod()) {
            $this->logger->warning("{action} method not allowed", $logContext);
            return $this->jsonResponse(405, [
                'errorMessage' => 'Method not allowed',
            ]);
        }

        try {
            /** @var Awardit_Altapay_Model_Type_NotificationCallback|null $body */
            $body = $this->getSerializer()->deserialize(
                $req->getRawBody(),
                'Awardit_Altapay_Model_Type_NotificationCallback',
                'json'
            );
        } catch (Throwable $e) {
            $this->logger->warning("{action} malformed request: {$e->getMessage()}", $logContext);
            return $this->jsonResponse(400, [
                'errorMessage' => 'Malformed request',
            ]);
        }

        if (empty($body) || empty($body->order) || empty($body->order['orderId'])) {
            $this->logger->warning("{action} missing order.orderId", $logContext);
            return $this->jsonResponse(400, [
                'errorMessage' => 'Malformed request',
            ]);
        }

        $logContext['orderId'] = $body->order['orderId'];

        $verified = $this->helper->verifySessionToken($body->sessionId, $body->order);
        if (!$verified) {
            $this->logger->error("Integrity check failed: {orderId}", $logContext);
            return $this->jsonResponse(400, [
                'errorMessage' => 'Unverified',
            ]);
        }

        $quote = $this->helper->getQuoteByQuoteId($body->order['paymentMetadata']['quoteId']);

        if (!$quote) {
            // Fallback, see if order has already been created
            $order = $this->helper->getOrderByOrderId($body->order['orderId']);
            if ($order) {
                return $this->jsonResponse(200, [
                    'result' => 'ok',
                ]);
            }

            $this->logger->warning("{action} quote not found: {orderId}", $logContext);
            return $this->jsonResponse(404, [
                'errorMessage' => 'Quote id not found',
            ]);
        }

        $payment = $quote->getPayment();
        /** @var Awardit_Altapay_Model_Method_Altapay */
        $method = $payment->getMethodInstance();

        // Create Magento order
        try {
            $method->createOrder($quote, $body);
            $this->logger->info("{action} verified order {orderId}", $logContext);
            return $this->jsonResponse(200, [
                'result' => 'ok',
            ]);
        } catch (Awardit_Altapay_Exception $e) {
            $this->logger->error("{action} error: {$e->getMessage()}", $logContext);
            return $this->jsonResponse($e->getCode(), [
                'errorMessage' => $e->getHttpMessage(),
            ]);
        } catch (Awardit_Antifraud_Exception $e) {
            $this->logger->error("{action} error: {$e->getMessage()}", $logContext);

            $ruleType = $e->getRuleType();
            if (in_array($ruleType, ["product_count", "product_value", "product_money"])) {
                $errorCode = MageQL_Sales_SubmitOrderException::FRAUD_PRODUCT;
            } elseif (in_array($ruleType, ["order_count", "order_value", "order_money", "item_count", "risk_score"])) {
                $errorCode = MageQL_Sales_SubmitOrderException::FRAUD_ORDER;
            } else {
                $errorCode = MageQL_Sales_SubmitOrderException::FRAUD_BLACKLIST;
            }

            return $this->jsonResponse(403, [
                'errorMessage' => $errorCode,
            ]);
        } catch (Throwable $e) {
            $this->logger->error("{action} error: {$e->getMessage()}", $logContext);
            return $this->jsonResponse(500, [
                'errorMessage' => 'Server error',
            ]);
        }
    }

    /**
     * Success callback from Altapay
     * @return Zend_Controller_Response_Abstract
     */
    public function successAction()
    {
        return $this->jsonResponse(200, [
            'result' => 'ok',
        ]);
    }
}
