<?php

/**
 * @psalm-type ColumnDefinition = array<array-key, mixed>
 * @psalm-type JoinContext = array<array-key, mixed>
 */
trait Awardit_AdminExtensions_Common_GridTrait
{
    /** @var array<array-key, ColumnDefinition> $columnDefinitions */
    private array $columnDefinitions = [];
    /** @var array<array-key, string> $columnSelected */
    private array $columnSelected = [];
    private array $columnDefaults = [];
    /** @var array<array-key, JoinContext> $joinContexts */
    private array $joinContexts = [];

    /**
     * Add context (conditional table join) enabler.
     * @param string $context Context identifier
     * @param string $resource Resource (table) identifier
     * @param string $join Join clause to use
     * @return $this
     */
    public function enableContext(string $context, string $resource, string $join, ?string $depends = null): self
    {
        $this->joinContexts[$context] = [
            'resource' => $resource,
            'join' => $join,
            'depends' => $depends,
            'keys' => [],
        ];
        return $this;
    }


    /* -------------- Must be added in implementing class ---------------------------- */

    // Add columns to be enabled by default
    abstract protected function addDefaults(): void;

    // Return name of underlying collection
    abstract protected function getCollectionClass(): string;

    // Return name of event to use in observer
    abstract protected function getEventName(): string;


    /* -------------- Interface methods ---------------------------------------------- */

    /**
     * Enable a column for column selection
     * @param ColumnDefinition $definition
     */
    public function enableColumn(string $key, array $definition, int $viewState = self::COL_OPTIONAL): void
    {
        if (array_key_exists('attribute', $definition) && method_exists($this->getCollection(), 'getAttribute')) {
            $definition['attribute'] = is_string($definition['attribute'])
                ? $this->getCollection()->getAttribute($definition['attribute'])
                : $definition['attribute'];
            $definition['header'] = $definition['header'] ?? $definition['attribute']->getFrontendLabel();
        }
        if ($viewState === self::COL_DEFAULT) {
            $this->columnDefaults[] = $key;
            $this->columnDefinitions[$key] = $definition;
        } elseif ($viewState === self::COL_STATIC) {
            $this->bindColumn($key, $definition);
        } else {
            $this->columnDefinitions[$key] = $definition;
        }
    }


    /* -------------- Grid methods --------------------------------------------------- */

    /**
     * @psalm-suppress PossiblyNullReference, PossiblyFalseReference
     */
    protected function prepareLayout(): void
    {
        $this->addDefaults();

        Mage::dispatchEvent($this->getEventName(), ['handler' => $this]);

        $helper = Mage::helper('awardit_adminextensions');
        $selected = $this->getSelectedColumns();
        $add = $rem = [];
        foreach ($this->getColumnDefinitions() as $code => $column) {
            if (in_array($code, $selected)) {
                $rem[] = ['value' => "rem.{$code}", 'label' => $column['header']];
            } else {
                $add[] = ['value' => "add.{$code}", 'label' => $column['header']];
            }
        }
        $this->setChild(
            'column_select',
            $this->getLayout()->createBlock('adminhtml/html_select')
                ->setOptions([
                    ['label' => $helper->__('Select columns'), 'value' => ''],
                    ['label' => $helper->__('Add column'), 'value' => $add],
                    ['label' => $helper->__('Remove column'), 'value' => $rem],
                ])->setExtraParams("onchange=\"selectColumn(this);\"")
        );
    }

    protected function _prepareLayout(): self
    {
        $this->prepareLayout();
        // Implement sort and some other things
        Mage_Adminhtml_Block_Widget_Grid::_prepareLayout();
        return $this;
    }

    /** @psalm-suppress UndefinedMethod */
    protected function prepareColumns(): void
    {
        $store = $this->getStore();
        if ($store && $this->getCollection() instanceof Mage_Eav_Model_Entity_Collection_Abstract) {
            $this->getCollection()->addStoreFilter($store);
        }
        $this->bindColumns();
    }

    protected function _prepareColumns(): self
    {
        $this->prepareColumns();
        // Implement sort and some other things
        Mage_Adminhtml_Block_Widget_Grid::_prepareColumns();
        return $this;
    }

    protected function prepareCollection(): void
    {
    }

    /** @psalm-suppress LessSpecificImplementedReturnType */
    protected function _prepareCollection(): self
    {
        $this->prepareCollection();
        // Implement sort and some other things
        Mage_Adminhtml_Block_Widget_Grid::_prepareCollection();
        return $this;
    }


    /* -------------- Grid UI methods ------------------------------------------------ */

    /**
     * Get the column selector.
     * @return string
     */
    public function getColumnSelectHtml(): string
    {
        return $this->getChildHtml('column_select');
    }

    /**
     * Get the main section (column selector and filter buttons).
     * @return string
     */
    public function getMainButtonsHtml(): string
    {
        if ($this->getHeadersVisibility() || $this->getFilterVisibility()) {
            return $this->getColumnSelectHtml() . parent::getMainButtonsHtml();
        }
        return '';
    }

    /**
     * Add JS code.
     * @param string $html
     * @return string
     */
    protected function _afterToHtml($html): string
    {
        return $html
            . '<script type="text/javascript">'
            . 'function selectColumn(elem) {'
            . "var ob = {$this->getJsObjectName()};"
            . 'ob.reload(ob.addVarToUrl("column", elem.value));'
            . '}'
            . '</script>';
    }


    /* -------------- Internal binder methods ---------------------------------------- */

    protected function bindColumns(): self
    {
        foreach ($this->getSelectedColumns() as $column) {
            $this->bindColumn($column);
            continue;
        }
        foreach ($this->joinContexts as $context => $definition) {
            $this->bindTable($context, $definition);
        }
        return $this;
    }

    /**
     * @param ColumnDefinition|null $definition
     * @psalm-suppress MissingClosureParamType
     */
    protected function bindColumn(string $key, ?array $definition = null): void
    {
        $definition = $definition ?? $this->columnDefinitions[$key] ?? null;
        if (!$definition) {
            return;
        }
        array_walk($definition, function (&$value, string $key) {
            $value = $key != 'callback' && is_callable($value) ? $value() : $value;
        });

        $store = $this->getStore();

        // Defaults
        $definition = array_merge([
            'index' => $key,
            'storeId' => $store ? $store->getId() : Mage_Core_Model_App::ADMIN_STORE_ID,
            'type' => 'text',
        ], $definition);

        // By implementation
        if (array_key_exists('attribute', $definition)) {
            $byAttribute = $this->bindAttribute($key, $definition);
            $definition = array_merge($byAttribute, $definition);
        } elseif (array_key_exists('context', $definition)) {
            $this->joinContexts[$definition['context']]['keys'][$key] = $definition['field'];
        } elseif (array_key_exists('table', $definition)) {
            $this->bindJoin($definition);
        }

        // Attach dependencies
        switch ($definition['type']) {
            case 'price':
                $definition['currency_code'] = $store ? $store->getBaseCurrency()->getCode() : '';
                break;
        }

        // Create column
        $this->addColumn($key, $definition);
        $column = $this->getColumn($key);

        array_walk($definition, function ($value, string $key) use ($column) {
            if ($key == 'callback' && is_callable($value)) {
                $value($column);
            }
        });
    }

    /**
     * @return ColumnDefinition
     * @param ColumnDefinition $definition
     * @psalm-suppress DocblockTypeContradiction, RedundantConditionGivenDocblockType
     */
    protected function bindAttribute(string $key, array $definition): array
    {
        $collection = $this->getCollection();
        if (!$collection instanceof Mage_Eav_Model_Entity_Collection_Abstract) {
            throw new LogicException('Not an EAV collection');
        }
        /** @var Mage_Catalog_Model_Resource_Eav_Attribute $attribute */
        $attribute = is_string($definition['attribute'])
            ? $collection->getAttribute($definition['attribute'])
            : $definition['attribute'];
        $required = $definition['required'] ?? $attribute->getIsRequired();
        $collection->joinAttribute(
            $definition['index'] ?? $key, // alias
            $attribute, // attribute name or class
            'entity_id', // bind
            null, // filter
            $required ? 'inner' : 'left', // joinType
            $definition['storeId'] ?? 0 // storeId
        );
        return [
            'header' => $attribute->getFrontendLabel(),
            'type' => $attribute->getFrontendInput(),
            'default' => $attribute->getDefaultValue(),
            'required' => $required,
        ];
    }

    /**
     * @param ColumnDefinition $definition
     * @psalm-suppress DocblockTypeContradiction, RedundantConditionGivenDocblockType
     */
    protected function bindJoin(array $definition): void
    {
        $collection = $this->getCollection();
        if (!$collection instanceof Mage_Eav_Model_Entity_Collection_Abstract) {
            throw new LogicException('Not an EAV collection');
        }
        $required = $definition['required'] ?? false;
        $collection->joinField(
            $definition['index'], // alias
            $definition['table'], // table
            $definition['field'], // field
            $definition['bind'], // bind
            $definition['cond'], // condition
            $required ? 'inner' : 'left', // joinType
        );
    }

    /**
     * @param JoinContext $definition
     * @psalm-suppress DocblockTypeContradiction, RedundantConditionGivenDocblockType, UndefinedMethod
     */
    protected function bindTable(string $context, array $definition): void
    {
        if (empty($definition['keys'])) {
            return;
        }
        $collection = $this->getCollection();
        if ($collection instanceof Mage_Core_Model_Resource_Db_Collection_Abstract) {
            if ($definition['depends']) {
                $dp = $this->joinContexts[$definition['depends']];
                $collection->leftJoin(
                    [$definition['depends'] => $dp['resource']],
                    $dp['join'],
                    $dp['keys']
                );
            }
            $collection->leftJoin(
                [$context => $definition['resource']],
                $definition['join'],
                $definition['keys']
            );
        } elseif ($collection instanceof Mage_Eav_Model_Entity_Collection_Abstract) {
            if ($definition['depends']) {
                $dp = $this->joinContexts[$definition['depends']];
                $collection->joinTable(
                    [$definition['depends'] => $dp['resource']],
                    $dp['join'],
                    $dp['keys'],
                    null,
                    'left'
                );
            }
            $collection->joinTable(
                [$context => $definition['resource']],
                $definition['join'],
                $definition['keys'],
                null,
                'left'
            );
        } else {
            throw new LogicException('Not a usable DB collection');
        }
    }


    /* -------------- Selection methods ---------------------------------------------- */

    protected function getStore(): ?Mage_Core_Model_Store
    {
        $storeId = (int) $this->getRequest()->getParam('store', 0);
        return Mage::app()->getStore($storeId);
    }

    /** @psalm-suppress DocblockTypeContradiction */
    public function getCollection()
    {
        if (parent::getCollection() === null) {
            $class = Mage::getResourceModel($this->getCollectionClass());
            if (!$class) {
                throw new LogicException("{$this->getCollectionClass()} is not a valid class");
            }
            parent::setCollection($class);
        }
        return parent::getCollection();
    }

    /**
     * Resolve columns seleceted for order view.
     * @return array<array-key, string>
     */
    protected function getSelectedColumns(): array
    {
        if (!empty($this->columnSelected)) {
            return array_unique($this->columnSelected);
        }

        $session = Mage::getSingleton('adminhtml/session');
        /** @var array<array-key, string> $columns */
        $columns = $session->getData("{$this->getId()}_columns") ?? $this->columnDefaults;

        // Add/remove by browser param
        if ($this->getRequest()->has('column')) {
            /** @var array{0: string, 1: string} $action */
            $action = explode('.', (string)$this->getRequest()->getParam('column'), 2);
            if ($action[0] == 'add' && !in_array($action[1], $columns)) {
                $columns[] = $action[1];
            } elseif ($action[0] == 'rem' && in_array($action[1], $columns)) {
                unset($columns[array_search($action[1], $columns)]);
            }
        }
        $session->setData("{$this->getId()}_columns", $columns);

        // @todo: If column referenced as filter, add it locally but not to session.

        $this->columnSelected = $columns;
        return array_unique($this->columnSelected);
    }

    /**
     * List available columns.
     * @return array<array-key, ColumnDefinition>
     */
    protected function getColumnDefinitions(): array
    {
        return $this->columnDefinitions;
    }
}
