<?php

/**
 * Override of admin order list.
 * Enables additional columns, and ability to configure which columns to show.
 * @psalm-suppress PropertyNotSetInConstructor
 */
class Awardit_AdminExtensions_Block_Order_Grid extends Mage_Adminhtml_Block_Sales_Order_Grid implements
    Awardit_AdminExtensions_Common_GridInterface
{
    use Awardit_AdminExtensions_Common_GridTrait;


    protected function getCollectionClass(): string
    {
        return 'awardit_adminextensions/order_gridCollection';
    }

    protected function getEventName(): string
    {
        return 'awardit_adminextensions_order_grid';
    }

    protected function addDefaults(): void
    {
        $gridHelper = Mage::helper('awardit_adminextensions');
        $salesHelper = Mage::helper('sales');

        $this->enableContext(
            'order',
            'sales/order',
            "(order.entity_id = main_table.entity_id)"
        );
        $this->enableContext(
            'shipping_address',
            'sales/order_address',
            "(shipping_address.entity_id = order.shipping_address_id)",
            'order'
        );
        $this->enableContext(
            'order_payment',
            'sales/order_payment',
            "(order_payment.parent_id = main_table.entity_id)",
        );

        $this->enableColumn('increment_id', [
            'header' => $salesHelper->__('Order #'),
            'escape' => true,
        ], self::COL_STATIC);
        $this->enableColumn('store_id', [
            'header' => $salesHelper->__('Store'),
            'type'  => 'options',
            'options' => function () use ($gridHelper) {
                return $gridHelper->getStoreOptions();
            },
        ], self::COL_DEFAULT);
        $this->enableColumn('created_at', [
            'header' => $salesHelper->__('Purchased On'),
            'type' => 'datetime',
        ], self::COL_DEFAULT);

        $this->enableColumn('created_at', [
            'header' => $salesHelper->__('Purchased On'),
            'type' => 'datetime',
        ], self::COL_DEFAULT);
        $this->enableColumn('updated_at', [
            'header' => $salesHelper->__('Updated On'),
            'type' => 'datetime',
        ], self::COL_DEFAULT);
        $this->enableColumn('billing_name', [
            'header' => $salesHelper->__('Bill to Name'),
        ], self::COL_DEFAULT);
        $this->enableColumn('shipping_name', [
            'header' => $salesHelper->__('Ship to Name'),
        ], self::COL_DEFAULT);
        $this->enableColumn('country_id', [
            'context' => 'shipping_address',
            'field' => 'country_id',
            'header' => $salesHelper->__('Ship to Country'),
            'type'  => 'options',
            'options' => function () use ($gridHelper) {
                return $gridHelper->getCountryOptions();
            },
        ], self::COL_DEFAULT);
        $this->enableColumn('base_grand_total', [
            'header' => $salesHelper->__('G.T. (Base)'),
            'type'  => 'currency',
            'filter_index' => 'main_table.base_grand_total',
            'currency' => function () {
                $this->joinContexts['order']['base_currency_code'] = 'base_currency_code';
                return 'base_currency_code';
            },
        ], self::COL_DEFAULT);
        $this->enableColumn('base_total_paid', [
            'header' => $salesHelper->__('Paid (Base)'),
            'type'  => 'currency',
            'filter_index' => 'main_table.base_total_paid',
            'currency' => function () {
                $this->joinContexts['order']['base_currency_code'] = 'base_currency_code';
                return 'base_currency_code';
            },
        ]);
        $this->enableColumn('base_currency_code', [
            'header' => $salesHelper->__('Currency (Base)'),
            'type'  => 'options',
            'filter_index' => 'main_table.base_currency_code',
            'options' => function () use ($gridHelper) {
                return $gridHelper->getCurrencyOptions();
            },
        ]);
        $this->enableColumn('grand_total', [
            'header' => $salesHelper->__('G.T. (Purchased)'),
            'type'  => 'currency',
            'currency' => function () {
                $this->joinContexts['order']['order_currency_code'] = 'order_currency_code';
                return 'order_currency_code';
            },
        ], self::COL_DEFAULT);
        $this->enableColumn('total_paid', [
            'header' => $salesHelper->__('Paid (Purchased)'),
            'type'  => 'currency',
            'filter_index' => 'main_table.total_paid',
            'currency' => function (): string {
                $this->joinContexts['order']['order_currency_code'] = 'order_currency_code';
                return 'order_currency_code';
            },
        ]);
        $this->enableColumn('order_currency_code', [
            'header' => $salesHelper->__('Currency (Purchased)'),
            'type'  => 'options',
            'options' => function () use ($gridHelper) {
                return $gridHelper->getCurrencyOptions();
            },
        ]);
        $this->enableColumn('status', [
            'header' => $salesHelper->__('Status'),
            'type'  => 'options',
            'filter_index' => 'main_table.status',
            'options' => function () use ($gridHelper) {
                return $gridHelper->getOrderStatusOptions();
            },
        ], self::COL_DEFAULT);
        $this->enableColumn('state', [
            'context' => 'order',
            'field' => 'state',
            'header' => $salesHelper->__('State (Order)'),
            'type'  => 'options',
            'options' => function () use ($gridHelper) {
                return $gridHelper->getOrderStateOptions();
            },
        ]);
        $this->enableColumn('shipping_method', [
            'context' => 'order',
            'field' => 'shipping_method',
            'header' => $salesHelper->__('Shipping'),
            'type'  => 'options',
            'renderer' => 'awardit_adminextensions/renderer_prefixOptions',
            'filter' => 'awardit_adminextensions/filter_prefixSelect',
            'options' => function () use ($gridHelper) {
                return $gridHelper->getShippingMethodOptions();
            },
        ]);
        $this->enableColumn('payment_method', [
            'context' => 'order_payment',
            'field' => 'method',
            'header' => $salesHelper->__('Payment'),
            'type'  => 'options',
            'filter_index' => 'order_payment.method',
            'column_index' => 'method',
            'callback' => function (Mage_Adminhtml_Block_Widget_Grid_Column $column) {
                $helper = Mage::helper('payment');
                $grouped_methods = [];
                $group_element = Mage::app()->getConfig()->getNode('global/payment/groups');
                if ($group_element === false) {
                    return;
                }
                /** @var array $groups */
                $groups = $group_element->asCanonicalArray();
                asort($groups);
                foreach ($groups as $code => $label) {
                    $grouped_methods[$code] = ['value' => [], 'label' => $label];
                }
                $grouped_methods['-'] = ['value' => [], 'label' => '-'];
                $methods = $helper->getPaymentMethodList();
                foreach ($methods as $code => $label) {
                    $class = $helper->getMethodModelClassName($code);
                    if (is_null($class)) {
                        continue;
                    }
                    if (empty($label)) {
                        continue;
                    }
                    /** @var object $instance */
                    $instance = Mage::getModel($class);
                    $group = $instance->getConfigData('group') ?: '-';
                    $grouped_methods[$group]['value'][$code] = ['value' => $code, 'label' => $label];
                }
                $column->setOptions($methods);
                $column->setOptionGroups($grouped_methods);
            },
        ]);
        $this->enableColumn('customer_email', [
            'context' => 'order',
            'field' => 'customer_email',
            'header' => $salesHelper->__('Email'),
        ]);
        $this->enableColumn('telephone', [
            'context' => 'shipping_address',
            'field' => 'telephone',
            'header' => $salesHelper->__('Phone'),
        ]);
        $this->enableColumn('remote_ip', [
            'context' => 'order',
            'field' => 'remote_ip',
            'header' => $salesHelper->__('IP-address'),
            'callback' => function (Mage_Adminhtml_Block_Widget_Grid_Column $column) {
                $this->joinContexts['order']['x_forwarded_for'] = 'x_forwarded_for';
                $column->setGetter(function (Varien_Object $row) {
                    if ($forwarded = $row->getData('x_forwarded_for')) {
                        return "{$row->getData('remote_ip')} ({$forwarded})";
                    }
                    return $row->getData('remote_ip');
                });
            },
        ]);
    }

    /**
     * Prepare columns.
     * @return $this
     * @psalm-suppress MoreSpecificReturnType, LessSpecificReturnStatement
     */
    protected function _prepareColumns(): self
    {
        $this->prepareColumns();

        // Render "action" column
        if (Mage::getSingleton('admin/session')->isAllowed('sales/order/actions/view')) {
            $this->addColumn('action', [
                'header' => Mage::helper('sales')->__('Action'),
                'width' => '50px',
                'type' => 'action',
                'getter' => 'getId',
                'actions' => [
                    [
                        'caption' => Mage::helper('sales')->__('View'),
                        'url' => array('base' => '*/sales_order/view'),
                        'field' => 'order_id',
                        'data-column' => 'action',
                    ]
                ],
                'filter' => false,
                'sortable' => false,
                'index' => 'stores',
                'is_system' => true,
            ]);
        }

        // Render some buttons
        $this->addRssList('rss/order/new', Mage::helper('sales')->__('New Order RSS'));
        $this->addExportType('*/*/exportCsv', Mage::helper('sales')->__('CSV'));
        $this->addExportType('*/*/exportExcel', Mage::helper('sales')->__('Excel XML'));

        return $this;
    }
}
