<?php

/**
 * Override of admin order list.
 * Enables additional columns, and ability to configure which columns to show.
 */
class Awardit_AdminExtensions_Block_Order_Grid extends Mage_Adminhtml_Block_Sales_Order_Grid
{
    /** @var array<string> */
    private $order_columns = [];

    /** @var array<array-key, array> */
    private $selected_columns;

    /** @var array<array-key, array> */
    private $available_columns;

    /**
     * List of available standard columns.
     * @return array<array-key, array>
     */
    protected function getStandardColumns(): array
    {
        return [
            'store_id' => [
                'header' => Mage::helper('sales')->__('Purchased From (Store)'),
                'default' => true,
            ],
            'created_at' => [
                'header' => Mage::helper('sales')->__('Purchased On'),
                'default' => true,
            ],
            'updated_at' => [
                'header' => Mage::helper('sales')->__('Updated On'),
                'default' => false,
            ],
            'billing_name' => [
                'header' => Mage::helper('sales')->__('Bill to Name'),
                'default' => true,
            ],
            'shipping_name' => [
                'header' => Mage::helper('sales')->__('Ship to Name'),
                'default' => true,
            ],
            'base_grand_total' => [
                'header' => Mage::helper('sales')->__('G.T. (Base)'),
                'default' => true,
            ],
            'base_total_paid' => [
                'header' => Mage::helper('sales')->__('Paid (Base)'),
                'default' => false,
            ],
            'base_currency_code' => [
                'header' => Mage::helper('sales')->__('Currency (Base)'),
                'default' => false,
            ],
            'grand_total' => [
                'header' => Mage::helper('sales')->__('G.T. (Purchased)'),
                'default' => true,
            ],
            'total_paid' => [
                'header' => Mage::helper('sales')->__('Paid (Purchased)'),
                'default' => false,
            ],
            'order_currency_code' => [
                'header' => Mage::helper('sales')->__('Currency (Purchased)'),
                'default' => false,
            ],
            'status' => [
                'header' => Mage::helper('sales')->__('Status'),
                'default' => true,
            ],
            'shipping_method' => [
                'header' => Mage::helper('sales')->__('Shipping'),
                'default' => false,
            ],
            'customer_email' => [
                'header' => Mage::helper('sales')->__('Email'),
                'default' => false,
            ],
            'remote_ip' => [
                'header' => Mage::helper('sales')->__('IP-address'),
                'default' => false,
            ],
        ];
    }

    /**
     * List of available columns.
     * @return array<array-key, array>
     */
    protected function getAvailableColumns(): array
    {
        if ($this->available_columns) {
            return $this->available_columns;
        }
        // @todo: Support registering additional columns from module
        return $this->available_columns = $this->getStandardColumns();
    }

    /**
     * Collection class to use as source.
     * @return string
     */
    protected function _getCollectionClass()
    {
        return 'awardit_adminextensions/order_gridCollection';
    }

    /**
     * Add JS code.
     * @return string
     */
    protected function _afterToHtml($html): string
    {
        return $html
            . '<script type="text/javascript">'
            . 'function selectColumn(elem) {'
            . "var ob = {$this->getJsObjectName()};"
            . 'ob.reload(ob.addVarToUrl("column", elem.value));'
            . '}'
            . '</script>';
    }

    /**
     * Get the column selector.
     * @return string
     */
    public function getColumnSelectHtml(): string
    {
        return $this->getChildHtml('column_select');
    }

    /**
     * Get the main section (column selector and filter buttons).
     * @return string
     */
    public function getMainButtonsHtml(): string
    {
        if ($this->getHeadersVisibility() || $this->getFilterVisibility()) {
            return $this->getColumnSelectHtml() . parent::getMainButtonsHtml();
        }
        return '';
    }

    /**
     * Resolve columns seleceted for order view.
     * @return array<string>
     */
    protected function getSelectedColumns(): array
    {
        if ($this->selected_columns) {
            return $this->selected_columns;
        }

        $session = Mage::getSingleton('adminhtml/session');
        $columns = $session->getData("{$this->getId()}_columns");

        // Add default columns if empty session
        if (is_null($columns)) {
            $columns = array_keys(array_filter($this->getAvailableColumns(), function ($column) {
                return $column['default'];
            }));
        }

        // Add/remove by browser param
        if ($this->getRequest()->has('column')) {
            $action = explode('.', $this->getRequest()->getParam('column'));
            if ($action[0] == 'add' && !in_array($action[1], $columns)) {
                $columns[] = $action[1];
            } elseif ($action[0] == 'rem' && in_array($action[1], $columns)) {
                unset($columns[array_search($action[1], $columns)]);
            }
        }
        $session->setData("{$this->getId()}_columns", $columns);

        // @todo: If column referenced as filter, add it locally but not to session.

        $this->selected_columns = $columns;

        return $this->selected_columns;
    }

    /**
     * Prepare layout.
     * @return this
     */
    protected function _prepareLayout(): self
    {
        $helper = Mage::helper('awardit_adminextensions');
        $selected = $this->getSelectedColumns();
        $add = $rem = [];

        foreach ($this->getAvailableColumns() as $code => $column) {
            if (in_array($code, $selected)) {
                $rem[] = ['value' => "rem.{$code}", 'label' => $column['header']];
            } else {
                $add[] = ['value' => "add.{$code}", 'label' => $column['header']];
            }
        }

        $this->setChild('column_select',
            $this->getLayout()->createBlock('adminhtml/html_select')
                ->setOptions([
                    ['label' => $helper->__('Select columns'), 'value' => ''],
                    ['label' => $helper->__('Add column'), 'value' => $add],
                    ['label' => $helper->__('Remove column'), 'value' => $rem],
                ])->setExtraParams("onchange=\"selectColumn(this);\"")
        );
        return parent::_prepareLayout();
    }

    /**
     * Prepare collection.
     * @return this
     */
    protected function _prepareCollection(): self
    {
        $collection = Mage::getResourceModel($this->_getCollectionClass());

        // Join order table if necessary.
        if (!empty($this->order_columns)) {
            $orders = $collection->getTable('sales/order');
            $select = $collection->getSelect();
            $select->joinLeft(
                ['source' => $orders],
                "(main_table.entity_id = source.entity_id)",
                array_unique($this->order_columns)
            );
        }
        $this->setCollection($collection);
        return Mage_Adminhtml_Block_Widget_Grid::_prepareCollection();
    }

    /**
     * Prepare columns.
     * @return this
     */
    protected function _prepareColumns(): self
    {
        $columns = $this->getAvailableColumns();

        // Always use # column
        $this->addColumn('real_order_id', array(
            'header' => Mage::helper('sales')->__('Order #'),
            'width'  => '80px',
            'type'   => 'text',
            'index'  => 'increment_id',
            'filter_index' => 'main_table.increment_id',
            'escape' => true,
        ));

        // Render specified columns
        foreach ($this->getSelectedColumns()  as $column) {
            switch ($column) {
                case 'store_id':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type' => 'store',
                        'store_view'=> true,
                        'display_deleted' => true,
                        'escape'  => true,
                    ]);
                    break;
                case 'created_at':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type' => 'datetime',
                        'width' => '100px',
                    ]);
                    break;
                case 'updated_at':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type' => 'datetime',
                        'width' => '100px',
                    ]);
                    break;
                case 'billing_name':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                    ]);
                    break;
                case 'shipping_name':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                    ]);
                    break;
                case 'base_grand_total':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type'  => 'currency',
                        'currency' => 'base_currency_code',
                    ]);
                    break;
                case 'base_total_paid':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type'  => 'currency',
                        'currency' => 'base_currency_code',
                    ]);
                    break;
                case 'base_currency_code':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type'  => 'options',
                        'width' => '70px',
                        'options' => $this->getCurrencies(),
                    ]);
                    break;
                case 'grand_total':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type'  => 'currency',
                        'currency' => 'order_currency_code',
                    ]);
                    break;
                case 'total_paid':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type'  => 'currency',
                        'currency' => 'order_currency_code',
                    ]);
                    break;
                case 'order_currency_code':
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type'  => 'options',
                        'width' => '70px',
                        'options' => $this->getCurrencies(),
                    ]);
                    break;
                case 'status':
                    $statuses = Mage::getSingleton('sales/order_config')->getStatuses();
                    asort($statuses);
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "main_table.{$column}",
                        'type'  => 'options',
                        'width' => '70px',
                        'options' => $statuses,
                    ]);
                    break;
                case 'shipping_method':
                    $methods = [];
                    foreach (Mage::getSingleton('shipping/config')->getAllCarriers() as $code => $carrier) {
                        if (Mage::getStoreConfig("carriers/{$code}/active")) {
                            $methods[$code] = Mage::getStoreConfig("carriers/{$code}/title");
                        }
                    }
                    asort($methods);
                    $this->order_columns[] = $column;
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "source.{$column}",
                        'type'  => 'options',
                        'width' => '70px',
                        'options' => $methods,
                        'renderer' => 'awardit_adminextensions/renderer_prefixOptions',
                        'filter' => 'awardit_adminextensions/filter_prefixSelect',
                    ]);
                    break;
                case 'customer_email':
                    $this->order_columns[] = $column;
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "source.{$column}",
                    ]);
                    break;
                case 'remote_ip':
                    $this->order_columns[] = $column;
                    $this->addColumn($column, [
                        'header' => $columns[$column]['header'],
                        'index' => $column,
                        'filter_index' => "source.{$column}",
                    ]);
                    break;
            }
        }

        if (Mage::getSingleton('admin/session')->isAllowed('sales/order/actions/view')) {
            $this->addColumn('action',
                array(
                    'header'    => Mage::helper('sales')->__('Action'),
                    'width'     => '50px',
                    'type'      => 'action',
                    'getter'     => 'getId',
                    'actions'   => array(
                        array(
                            'caption' => Mage::helper('sales')->__('View'),
                            'url'     => array('base'=>'*/sales_order/view'),
                            'field'   => 'order_id',
                            'data-column' => 'action',
                        )
                    ),
                    'filter'    => false,
                    'sortable'  => false,
                    'index'     => 'stores',
                    'is_system' => true,
            ));
        }
        $this->addRssList('rss/order/new', Mage::helper('sales')->__('New Order RSS'));

        $this->addExportType('*/*/exportCsv', Mage::helper('sales')->__('CSV'));
        $this->addExportType('*/*/exportExcel', Mage::helper('sales')->__('Excel XML'));

        return Mage_Adminhtml_Block_Widget_Grid::_prepareColumns();
    }

    /**
     * Get currencies for option field.
     * @return this
     */
    private function getCurrencies(): array
    {
        $currencies = Mage::getModel('directory/currency')->getConfigAllowCurrencies();
        sort($currencies);
        return array_combine($currencies, $currencies);
    }
}
