<?php

/**
 * Override of admin order list.
 * Enables additional columns, and ability to configure which columns to show.
 * @psalm-suppress PropertyNotSetInConstructor
 */
class Awardit_AdminExtensions_Block_Order_Grid extends Mage_Adminhtml_Block_Sales_Order_Grid
{
    /** @var array{column: null, context: null, header: null} $required_column_fields */
    private static $required_column_fields = [
        'context' => null,
        'column' => null,
        'header' => null
    ];

    /** @var array|null */
    private $selected_columns = null;

    /** @var array<array-key, array> */
    private $available_columns = [];

    /** @var array<array-key, Closure|null> */
    private $available_callbacks = [];

    /** @var array<array-key, array> */
    private $context_joins = [];

    /** @var array<array-key, array> */
    private $context_columns = ['main_table' => [
        'entity_id' => 'entity_id',
        'increment_id' => 'increment_id',
    ]];


    /* -------------- Public interface methods --------------------------------------- */

    /**
     * Add context (conditional table join) enabler.
     * @param string $context Context identifier
     * @param string $resource Resource (table) identifier
     * @param mixed $join Join clause to use
     * @return $this
     */
    public function enableContext(string $context, string $resource, $join = null): self
    {
        $this->context_joins[$context] = [
            'resource' => $resource,
            'join' => $join,
        ];
        return $this;
    }

    /**
     * Add column enabler.
     * @param array $column Column definition as associative array
     * @param Closure|null $callback Optional callback
     * @return $this
     */
    public function enableColumn(array $column, ?Closure $callback = null): self
    {
        $match = array_intersect_key($column, self::$required_column_fields);
        if (count($match) != 3) {
            $diff = array_diff_key(self::$required_column_fields, $match);
            Mage::log(sprintf(
                'Invalid column definition, missing fields %s on definition %s',
                json_encode(array_keys($diff)),
                json_encode($column)
            ), Zend_Log::WARN);
            return $this; // Skip invalid
        }
        $column = array_merge([
            'default' => false,
        ], $column);
        $this->available_columns[$column['column']] = $column;
        $this->available_callbacks[$column['column']] = $callback;
        return $this;
    }


    /* -------------- Internal handler methods --------------------------------------- */

    /**
     * Adds default context and column enablers.
     */
    protected function addDefaults(): void
    {
        $helper = Mage::helper('sales');

        $this->enableContext(
            'order_payment',
            'sales/order_payment',
            "(order_payment.parent_id = main_table.entity_id)"
        );
        $this->enableContext(
            'order',
            'sales/order',
            "(order.entity_id = main_table.entity_id)"
        );
        $this->enableContext(
            'shipping_address',
            'sales/order_address',
            "(shipping_address.entity_id = order.shipping_address_id)"
        );
        $this->enableContext(
            'store',
            'core/store',
            "(store.store_id = main_table.store_id)"
        );

        $this->enableColumn([
            'context' => 'store',
            'column' => 'name',
            'header' => $helper->__('Store'),
            'default' => true,
            'type'  => 'options',
            'filter_index' => 'store.store_id',
        ], function (string $column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $stores = [];
            foreach (Mage::app()->getStores() as $store) {
                $stores[$store->getId()] = $store->getName();
            }
            asort($stores);
            $object->setOptions($stores);
        });
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'created_at',
            'header' => $helper->__('Purchased On'),
            'default' => true,
            'type' => 'datetime',
        ]);
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'updated_at',
            'header' => $helper->__('Updated On'),
            'type' => 'datetime',
        ]);
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'billing_name',
            'header' => $helper->__('Bill to Name'),
            'default' => true,
        ]);
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'shipping_name',
            'header' => $helper->__('Ship to Name'),
            'default' => true,
        ]);
        $this->enableColumn([
            'context' => 'shipping_address',
            'column' => 'country_id',
            'header' => $helper->__('Ship to Country'),
            'type'  => 'options',
        ], function (string $column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $object->setOptions($this->getCountries());
        });
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'base_grand_total',
            'header' => $helper->__('G.T. (Base)'),
            'default' => true,
            'type'  => 'currency',
            'currency' => 'base_currency_code',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $_object) {
            $this->context_columns['order']['base_currency_code'] = 'base_currency_code';
        });
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'base_total_paid',
            'header' => $helper->__('Paid (Base)'),
            'type'  => 'currency',
            'currency' => 'base_currency_code',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $_object) {
            $this->context_columns['order']['base_currency_code'] = 'base_currency_code';
        });
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'base_currency_code',
            'header' => $helper->__('Currency (Base)'),
            'type'  => 'options',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $object->setOptions($this->getCurrencies());
        });
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'grand_total',
            'header' => $helper->__('G.T. (Purchased)'),
            'default' => true,
            'type'  => 'currency',
            'currency' => 'order_currency_code',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $_object) {
            $this->context_columns['order']['order_currency_code'] = 'order_currency_code';
        });
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'total_paid',
            'header' => $helper->__('Paid (Purchased)'),
            'type'  => 'currency',
            'currency' => 'order_currency_code',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $_object) {
            $this->context_columns['order']['order_currency_code'] = 'order_currency_code';
        });
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'order_currency_code',
            'header' => $helper->__('Currency (Purchased)'),
            'type'  => 'options',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $object->setOptions($this->getCurrencies());
        });
        $this->enableColumn([
            'context' => 'main_table',
            'column' => 'status',
            'header' => $helper->__('Status'),
            'default' => true,
            'type'  => 'options',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $statuses = Mage::getSingleton('sales/order_config')->getStatuses();
            asort($statuses);
            $object->setOptions($statuses);
        });
        $this->enableColumn([
            'context' => 'order',
            'column' => 'state',
            'header' => $helper->__('State (Order)'),
            'type'  => 'options',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $states = Mage::getSingleton('sales/order_config')->getStates();
            asort($states);
            $object->setOptions($states);
        });
        $this->enableColumn([
            'context' => 'order',
            'column' => 'shipping_method',
            'header' => $helper->__('Shipping'),
            'type'  => 'options',
            'renderer' => 'awardit_adminextensions/renderer_prefixOptions',
            'filter' => 'awardit_adminextensions/filter_prefixSelect',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $methods = [];
            foreach (Mage::getSingleton('shipping/config')->getAllCarriers() as $code => $_carrier) {
                if (Mage::getStoreConfig("carriers/{$code}/active")) {
                    $methods[$code] = Mage::getStoreConfig("carriers/{$code}/title");
                }
            }
            asort($methods);
            $object->setOptions($methods);
        });
        $this->enableColumn([
            'context' => 'order_payment',
            'column' => 'payment_method',
            'header' => $helper->__('Payment'),
            'type'  => 'options',
            'filter_index' => 'order_payment.method',
            'column_index' => 'method',
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $helper = Mage::helper('payment');
            $grouped_methods = [];
            /** @var Mage_Core_Model_Config_Element $group_element */
            $group_element = Mage::app()->getConfig()->getNode('global/payment/groups');
            /** @var array $groups */
            $groups = $group_element->asCanonicalArray();
            asort($groups);
            foreach ($groups as $code => $label) {
                $grouped_methods[$code] = ['value' => [], 'label' => $label];
            }
            $grouped_methods['-'] = ['value' => [], 'label' => '-'];
            $methods = $helper->getPaymentMethodList();
            foreach ($methods as $code => $label) {
                if (!$instance = $helper->getMethodInstance($code)) {
                    continue;
                }
                if (empty($label)) {
                    continue;
                }
                $group = $instance->getConfigData('group') ?: '-';
                $grouped_methods[$group]['value'][$code] = ['value' => $code, 'label' => $label];
            }
            $object->setOptions($methods);
            $object->setOptionGroups($grouped_methods);
        });
        $this->enableColumn([
            'context' => 'order',
            'column' => 'customer_email',
            'header' => $helper->__('Email'),
        ]);
        $this->enableColumn([
            'context' => 'order',
            'column' => 'remote_ip',
            'header' => $helper->__('IP-address'),
            'filter_index' => ['order.remote_ip', 'order.x_forwarded_for'],
        ], function (string $_column, Mage_Adminhtml_Block_Widget_Grid_Column $object) {
            $this->context_columns['order']['x_forwarded_for'] = 'x_forwarded_for';
            $object->setGetter(function (Varien_Object $row) {
                if ($forwarded = $row->getData('x_forwarded_for')) {
                    return "{$row->getData('remote_ip')} ({$forwarded})";
                }
                return $row->getData('remote_ip');
            });
        });
    }

    /**
     * List available columns.
     * @return array<array-key, array>
     */
    protected function getAvailableColumns(): array
    {
        return $this->available_columns;
    }

    /**
     * Collection class to use as source.
     * @return string
     */
    protected function _getCollectionClass(): string
    {
        return 'awardit_adminextensions/order_gridCollection';
    }

    /**
     * Add filter. Contrary to overloaded method, this support multiple columns.
     * @param Mage_Adminhtml_Block_Widget_Grid_Column $column
     * @return $this
     * @psalm-suppress RedundantConditionGivenDocblockType, PossiblyFalseReference, NoValue
     */
    protected function _addColumnFilterToCollection($column): self
    {
        if ($this->getCollection()) {
            /** @var array|string $field */
            $field = $column->getFilterIndex() ?: $column->getIndex();
            if ($column->getFilterConditionCallback() && $column->getFilterConditionCallback()[0] instanceof self) {
                call_user_func($column->getFilterConditionCallback(), $this->getCollection(), $column);
            } else {
                $cond = $column->getFilter()->getCondition();
                if (is_array($field)) {
                    $cond = array_fill(0, count($field), $cond);
                }
                if ($field && $cond) {
                    /** @var Varien_Data_Collection_Db $collection */
                    $collection = $this->getCollection();
                    $collection->addFieldToFilter($field, $cond);
                }
            }
        }
        return $this;
    }

    /**
     * Add JS code.
     * @param string $html
     * @return string
     */
    protected function _afterToHtml($html): string
    {
        return $html
            . '<script type="text/javascript">'
            . 'function selectColumn(elem) {'
            . "var ob = {$this->getJsObjectName()};"
            . 'ob.reload(ob.addVarToUrl("column", elem.value));'
            . '}'
            . '</script>';
    }

    /**
     * Get the column selector.
     * @return string
     */
    public function getColumnSelectHtml(): string
    {
        return $this->getChildHtml('column_select');
    }

    /**
     * Get the main section (column selector and filter buttons).
     * @return string
     */
    public function getMainButtonsHtml(): string
    {
        if ($this->getHeadersVisibility() || $this->getFilterVisibility()) {
            return $this->getColumnSelectHtml() . parent::getMainButtonsHtml();
        }
        return '';
    }

    /**
     * Resolve columns seleceted for order view.
     * @return array
     */
    protected function getSelectedColumns(): array
    {
        if (!is_null($this->selected_columns)) {
            return $this->selected_columns;
        }

        $session = Mage::getSingleton('adminhtml/session');
        $columns = $session->getData("{$this->getId()}_columns");

        // Add default columns if empty session
        if (is_null($columns)) {
            $columns = array_keys(array_filter($this->getAvailableColumns(), function ($column): bool {
                return $column['default'];
            }));
        }

        // Add/remove by browser param
        if ($this->getRequest()->has('column')) {
            /** @var array{0: string, 1: string} $action */
            $action = explode('.', $this->getRequest()->getParam('column'), 2);
            if ($action[0] == 'add' && !in_array($action[1], $columns)) {
                $columns[] = $action[1];
            } elseif ($action[0] == 'rem' && in_array($action[1], $columns)) {
                unset($columns[array_search($action[1], $columns)]);
            }
        }
        $session->setData("{$this->getId()}_columns", $columns);

        // @todo: If column referenced as filter, add it locally but not to session.

        $this->selected_columns = $columns;
        return $this->selected_columns;
    }

    /**
     * Prepare layout.
     * @return $this
     * @psalm-suppress PossiblyNullReference, PossiblyFalseReference
     */
    protected function _prepareLayout(): self
    {
        $this->addDefaults();

        Mage::dispatchEvent('awardit_adminextensions_order_grid', ['handler' => $this]);

        $helper = Mage::helper('awardit_adminextensions');
        $selected = $this->getSelectedColumns();
        $add = $rem = [];

        foreach ($this->getAvailableColumns() as $code => $column) {
            if (in_array($code, $selected)) {
                $rem[] = ['value' => "rem.{$code}", 'label' => $column['header']];
            } else {
                $add[] = ['value' => "add.{$code}", 'label' => $column['header']];
            }
        }

        $this->setChild(
            'column_select',
            $this->getLayout()->createBlock('adminhtml/html_select')
                ->setOptions([
                    ['label' => $helper->__('Select columns'), 'value' => ''],
                    ['label' => $helper->__('Add column'), 'value' => $add],
                    ['label' => $helper->__('Remove column'), 'value' => $rem],
                ])->setExtraParams("onchange=\"selectColumn(this);\"")
        );
        return parent::_prepareLayout();
    }

    /**
     * Prepare collection.
     * @return $this
     * @psalm-suppress MoreSpecificReturnType, LessSpecificReturnStatement
     */
    protected function _prepareCollection(): self
    {
        /** @var Awardit_AdminExtensions_Model_Resource_Order_GridCollection $collection */
        $collection = Mage::getResourceModel($this->_getCollectionClass());
        $select = $collection->getSelect();
        $select->reset(Zend_Db_Select::COLUMNS);
        $select->columns($this->context_columns['main_table']);

        // Join tables, if used
        foreach ($this->context_joins as $context => $info) {
            if (!isset($this->context_columns[$context])) {
                continue; // Table not used, skip it
            }
            $table = $collection->getTable($info['resource']);
            $select->joinLeft(
                [$context => $table],
                $info['join'],
                $this->context_columns[$context]
            );
        }

        $this->setCollection($collection);

        return Mage_Adminhtml_Block_Widget_Grid::_prepareCollection();
    }

    /**
     * Prepare columns.
     * @return $this
     * @psalm-suppress MoreSpecificReturnType, LessSpecificReturnStatement
     */
    protected function _prepareColumns(): self
    {
        $columns = $this->getAvailableColumns();

        // Always use # column
        $this->addColumn('real_order_id', array(
            'header' => Mage::helper('sales')->__('Order #'),
            'type'   => 'text',
            'index'  => 'increment_id',
            'filter_index' => 'main_table.increment_id',
            'escape' => true,
        ));

        // Render specified columns
        foreach ($this->getSelectedColumns() as $column) {
            if (!isset($columns[$column])) {
                Mage::log("Column '{$column}' has no definition.", Zend_Log::WARN);
                continue; // Can not render this
            }
            $source = $columns[$column];
            $source = array_merge([
                'default' => false,
                'index' => $source['column'],
                'filter_index' => "{$source['context']}.{$source['column']}",
                'column_index' => "{$source['context']}.{$source['column']}",
            ], $source);
            $this->addColumn($column, $source);
            $cb = $this->available_callbacks[$column];
            if (is_callable($cb)) {
                call_user_func($cb, $column, $this->getColumn($column));
            }
            $this->context_columns[$source['context']][$column] = $source['column_index'];
        }

        // Render "action" column
        if (Mage::getSingleton('admin/session')->isAllowed('sales/order/actions/view')) {
            $this->addColumn('action', [
                'header' => Mage::helper('sales')->__('Action'),
                'width' => '50px',
                'type' => 'action',
                'getter' => 'getId',
                'actions' => [
                    [
                        'caption' => Mage::helper('sales')->__('View'),
                        'url' => array('base' => '*/sales_order/view'),
                        'field' => 'order_id',
                        'data-column' => 'action',
                    ]
                ],
                'filter' => false,
                'sortable' => false,
                'index' => 'stores',
                'is_system' => true,
            ]);
        }

        // Render some buttons
        $this->addRssList('rss/order/new', Mage::helper('sales')->__('New Order RSS'));
        $this->addExportType('*/*/exportCsv', Mage::helper('sales')->__('CSV'));
        $this->addExportType('*/*/exportExcel', Mage::helper('sales')->__('Excel XML'));
        return Mage_Adminhtml_Block_Widget_Grid::_prepareColumns();
    }

    /**
     * Get currencies for option field.
     * @return array<array-key, mixed>
     */
    private function getCurrencies(): array
    {
        $currencies = Mage::getModel('directory/currency')->getConfigAllowCurrencies();
        sort($currencies);
        return array_combine($currencies, $currencies);
    }

    /**
     * Get countries for option field.
     * @return array<array-key, mixed>
     */
    private function getCountries(): array
    {
        $countries = [];
        foreach (Mage::helper('directory')->getCountryCollection() as $code => $country) {
            $countries[$code] = $country->getName();
        }
        return $countries;
    }
}
